(function() {
  var AtomReporter, SpecResultView, SuiteResultView, _, formatStackTrace, grim, ipcHelpers, listen, marked, path, process;

  path = require('path');

  process = require('process');

  _ = require('underscore-plus');

  grim = require('grim');

  marked = require('marked');

  listen = require('../src/delegated-listener');

  ipcHelpers = require('../src/ipc-helpers');

  formatStackTrace = function(spec, message, stackTrace) {
    var errorMatch, firstJasmineLinePattern, i, index, j, jasminePattern, len, len1, line, lines, prefixMatch, ref, ref1, ref2;
    if (message == null) {
      message = '';
    }
    if (!stackTrace) {
      return stackTrace;
    }
    jasminePattern = /^\s*at\s+.*\(?.*[\/\\]jasmine(-[^\/\\]*)?\.js:\d+:\d+\)?\s*$/;
    firstJasmineLinePattern = /^\s*at [\/\\].*[\/\\]jasmine(-[^\/\\]*)?\.js:\d+:\d+\)?\s*$/;
    lines = [];
    ref = stackTrace.split('\n');
    for (i = 0, len = ref.length; i < len; i++) {
      line = ref[i];
      if (!jasminePattern.test(line)) {
        lines.push(line);
      }
      if (firstJasmineLinePattern.test(line)) {
        break;
      }
    }
    errorMatch = (ref1 = lines[0]) != null ? ref1.match(/^Error: (.*)/) : void 0;
    if (message.trim() === (errorMatch != null ? (ref2 = errorMatch[1]) != null ? ref2.trim() : void 0 : void 0)) {
      lines.shift();
    }
    for (index = j = 0, len1 = lines.length; j < len1; index = ++j) {
      line = lines[index];
      prefixMatch = line.match(/at jasmine\.Spec\.<anonymous> \(([^)]+)\)/);
      if (prefixMatch) {
        line = "at " + prefixMatch[1];
      }
      if (process.platform === 'win32') {
        line = line.replace('file:///', '').replace(RegExp("" + path.posix.sep, "g"), path.win32.sep);
      }
      line = line.replace("at " + spec.specDirectory + path.sep, 'at ');
      lines[index] = line.replace("(" + spec.specDirectory + path.sep, '(');
    }
    lines = lines.map(function(line) {
      return line.trim();
    });
    return lines.join('\n').trim();
  };

  module.exports = AtomReporter = (function() {
    function AtomReporter() {
      var element, i, len, ref;
      this.element = document.createElement('div');
      this.element.classList.add('spec-reporter-container');
      this.element.innerHTML = "<div class=\"spec-reporter\">\n  <div class=\"padded pull-right\">\n    <button outlet=\"reloadButton\" class=\"btn btn-small reload-button\">Reload Specs</button>\n  </div>\n  <div outlet=\"coreArea\" class=\"symbol-area\">\n    <div outlet=\"coreHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"coreSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"bundledArea\" class=\"symbol-area\">\n    <div outlet=\"bundledHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"bundledSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"userArea\" class=\"symbol-area\">\n    <div outlet=\"userHeader\" class=\"symbol-header\"></div>\n    <ul outlet=\"userSummary\"class=\"symbol-summary list-unstyled\"></ul>\n  </div>\n  <div outlet=\"status\" class=\"status alert alert-info\">\n    <div outlet=\"time\" class=\"time\"></div>\n    <div outlet=\"specCount\" class=\"spec-count\"></div>\n    <div outlet=\"message\" class=\"message\"></div>\n  </div>\n  <div outlet=\"results\" class=\"results\"></div>\n  <div outlet=\"deprecations\" class=\"status alert alert-warning\" style=\"display: none\">\n    <span outlet=\"deprecationStatus\">0 deprecations</span>\n    <div class=\"deprecation-toggle\"></div>\n  </div>\n  <div outlet=\"deprecationList\" class=\"deprecation-list\"></div>\n</div>";
      ref = this.element.querySelectorAll('[outlet]');
      for (i = 0, len = ref.length; i < len; i++) {
        element = ref[i];
        this[element.getAttribute('outlet')] = element;
      }
    }

    AtomReporter.prototype.startedAt = null;

    AtomReporter.prototype.runningSpecCount = 0;

    AtomReporter.prototype.completeSpecCount = 0;

    AtomReporter.prototype.passedCount = 0;

    AtomReporter.prototype.failedCount = 0;

    AtomReporter.prototype.skippedCount = 0;

    AtomReporter.prototype.totalSpecCount = 0;

    AtomReporter.prototype.deprecationCount = 0;

    AtomReporter.timeoutId = 0;

    AtomReporter.prototype.reportRunnerStarting = function(runner) {
      var specs;
      this.handleEvents();
      this.startedAt = Date.now();
      specs = runner.specs();
      this.totalSpecCount = specs.length;
      this.addSpecs(specs);
      return document.body.appendChild(this.element);
    };

    AtomReporter.prototype.reportRunnerResults = function(runner) {
      this.updateSpecCounts();
      if (this.failedCount === 0) {
        this.status.classList.add('alert-success');
        this.status.classList.remove('alert-info');
      }
      if (this.failedCount === 1) {
        return this.message.textContent = this.failedCount + " failure";
      } else {
        return this.message.textContent = this.failedCount + " failures";
      }
    };

    AtomReporter.prototype.reportSuiteResults = function(suite) {};

    AtomReporter.prototype.reportSpecResults = function(spec) {
      this.completeSpecCount++;
      spec.endedAt = Date.now();
      this.specComplete(spec);
      return this.updateStatusView(spec);
    };

    AtomReporter.prototype.reportSpecStarting = function(spec) {
      return this.specStarted(spec);
    };

    AtomReporter.prototype.handleEvents = function() {
      listen(document, 'click', '.spec-toggle', function(event) {
        var specFailures;
        specFailures = event.currentTarget.parentElement.querySelector('.spec-failures');
        if (specFailures.style.display === 'none') {
          specFailures.style.display = '';
          event.currentTarget.classList.remove('folded');
        } else {
          specFailures.style.display = 'none';
          event.currentTarget.classList.add('folded');
        }
        return event.preventDefault();
      });
      listen(document, 'click', '.deprecation-list', function(event) {
        var deprecationList;
        deprecationList = event.currentTarget.parentElement.querySelector('.deprecation-list');
        if (deprecationList.style.display === 'none') {
          deprecationList.style.display = '';
          event.currentTarget.classList.remove('folded');
        } else {
          deprecationList.style.display = 'none';
          event.currentTarget.classList.add('folded');
        }
        return event.preventDefault();
      });
      listen(document, 'click', '.stack-trace', function(event) {
        return event.currentTarget.classList.toggle('expanded');
      });
      return this.reloadButton.addEventListener('click', function() {
        return ipcHelpers.call('window-method', 'reload');
      });
    };

    AtomReporter.prototype.updateSpecCounts = function() {
      var specCount;
      if (this.skippedCount) {
        specCount = (this.completeSpecCount - this.skippedCount) + "/" + (this.totalSpecCount - this.skippedCount) + " (" + this.skippedCount + " skipped)";
      } else {
        specCount = this.completeSpecCount + "/" + this.totalSpecCount;
      }
      return this.specCount.textContent = specCount;
    };

    AtomReporter.prototype.updateStatusView = function(spec) {
      var rootSuite, time;
      if (this.failedCount > 0) {
        this.status.classList.add('alert-danger');
        this.status.classList.remove('alert-info');
      }
      this.updateSpecCounts();
      rootSuite = spec.suite;
      while (rootSuite.parentSuite) {
        rootSuite = rootSuite.parentSuite;
      }
      this.message.textContent = rootSuite.description;
      time = "" + (Math.round((spec.endedAt - this.startedAt) / 10));
      if (time.length < 3) {
        time = "0" + time;
      }
      return this.time.textContent = time.slice(0, -2) + "." + time.slice(-2) + "s";
    };

    AtomReporter.prototype.specTitle = function(spec) {
      var desc, i, indent, len, parentDescs, s, suiteString;
      parentDescs = [];
      s = spec.suite;
      while (s) {
        parentDescs.unshift(s.description);
        s = s.parentSuite;
      }
      suiteString = "";
      indent = "";
      for (i = 0, len = parentDescs.length; i < len; i++) {
        desc = parentDescs[i];
        suiteString += indent + desc + "\n";
        indent += "  ";
      }
      return suiteString + " " + indent + " it " + spec.description;
    };

    AtomReporter.prototype.addSpecs = function(specs) {
      var bundledPackageSpecs, coreSpecs, i, len, packageFolderName, packageName, spec, specDirectory, symbol, userPackageSpecs;
      coreSpecs = 0;
      bundledPackageSpecs = 0;
      userPackageSpecs = 0;
      for (i = 0, len = specs.length; i < len; i++) {
        spec = specs[i];
        symbol = document.createElement('li');
        symbol.setAttribute('id', "spec-summary-" + spec.id);
        symbol.setAttribute('title', this.specTitle(spec));
        symbol.className = "spec-summary pending";
        switch (spec.specType) {
          case 'core':
            coreSpecs++;
            this.coreSummary.appendChild(symbol);
            break;
          case 'bundled':
            bundledPackageSpecs++;
            this.bundledSummary.appendChild(symbol);
            break;
          case 'user':
            userPackageSpecs++;
            this.userSummary.appendChild(symbol);
        }
      }
      if (coreSpecs > 0) {
        this.coreHeader.textContent = "Core Specs (" + coreSpecs + ")";
      } else {
        this.coreArea.style.display = 'none';
      }
      if (bundledPackageSpecs > 0) {
        this.bundledHeader.textContent = "Bundled Package Specs (" + bundledPackageSpecs + ")";
      } else {
        this.bundledArea.style.display = 'none';
      }
      if (userPackageSpecs > 0) {
        if (coreSpecs === 0 && bundledPackageSpecs === 0) {
          specDirectory = specs[0].specDirectory;
          packageFolderName = path.basename(path.dirname(specDirectory));
          packageName = _.undasherize(_.uncamelcase(packageFolderName));
          return this.userHeader.textContent = packageName + " Specs";
        } else {
          return this.userHeader.textContent = "User Package Specs (" + userPackageSpecs + ")";
        }
      } else {
        return this.userArea.style.display = 'none';
      }
    };

    AtomReporter.prototype.specStarted = function(spec) {
      return this.runningSpecCount++;
    };

    AtomReporter.prototype.specComplete = function(spec) {
      var results, specSummaryElement, specView;
      specSummaryElement = document.getElementById("spec-summary-" + spec.id);
      specSummaryElement.classList.remove('pending');
      results = spec.results();
      if (results.skipped) {
        specSummaryElement.classList.add("skipped");
        return this.skippedCount++;
      } else if (results.passed()) {
        specSummaryElement.classList.add("passed");
        return this.passedCount++;
      } else {
        specSummaryElement.classList.add("failed");
        specView = new SpecResultView(spec);
        specView.attach();
        return this.failedCount++;
      }
    };

    return AtomReporter;

  })();

  SuiteResultView = (function() {
    function SuiteResultView(suite1) {
      this.suite = suite1;
      this.element = document.createElement('div');
      this.element.className = 'suite';
      this.element.setAttribute('id', "suite-view-" + this.suite.id);
      this.description = document.createElement('div');
      this.description.className = 'description';
      this.description.textContent = this.suite.description;
      this.element.appendChild(this.description);
    }

    SuiteResultView.prototype.attach = function() {
      return (this.parentSuiteView() || document.querySelector('.results')).appendChild(this.element);
    };

    SuiteResultView.prototype.parentSuiteView = function() {
      var suiteView, suiteViewElement;
      if (!this.suite.parentSuite) {
        return;
      }
      if (!(suiteViewElement = document.querySelector("#suite-view-" + this.suite.parentSuite.id))) {
        suiteView = new SuiteResultView(this.suite.parentSuite);
        suiteView.attach();
        suiteViewElement = suiteView.element;
      }
      return suiteViewElement;
    };

    return SuiteResultView;

  })();

  SpecResultView = (function() {
    function SpecResultView(spec1) {
      var description, i, len, ref, result, resultElement, stackTrace, traceElement;
      this.spec = spec1;
      this.element = document.createElement('div');
      this.element.className = 'spec';
      this.element.innerHTML = "<div class='spec-toggle'></div>\n<div outlet='description' class='description'></div>\n<div outlet='specFailures' class='spec-failures'></div>";
      this.description = this.element.querySelector('[outlet="description"]');
      this.specFailures = this.element.querySelector('[outlet="specFailures"]');
      this.element.classList.add("spec-view-" + this.spec.id);
      description = this.spec.description;
      if (description.indexOf('it ') !== 0) {
        description = "it " + description;
      }
      this.description.textContent = description;
      ref = this.spec.results().getItems();
      for (i = 0, len = ref.length; i < len; i++) {
        result = ref[i];
        if (!(!result.passed())) {
          continue;
        }
        stackTrace = formatStackTrace(this.spec, result.message, result.trace.stack);
        resultElement = document.createElement('div');
        resultElement.className = 'result-message fail';
        resultElement.textContent = result.message;
        this.specFailures.appendChild(resultElement);
        if (stackTrace) {
          traceElement = document.createElement('pre');
          traceElement.className = 'stack-trace padded';
          traceElement.textContent = stackTrace;
          this.specFailures.appendChild(traceElement);
        }
      }
    }

    SpecResultView.prototype.attach = function() {
      return this.parentSuiteView().appendChild(this.element);
    };

    SpecResultView.prototype.parentSuiteView = function() {
      var suiteView, suiteViewElement;
      if (!(suiteViewElement = document.querySelector("#suite-view-" + this.spec.suite.id))) {
        suiteView = new SuiteResultView(this.spec.suite);
        suiteView.attach();
        suiteViewElement = suiteView.element;
      }
      return suiteViewElement;
    };

    return SpecResultView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
