(function() {
  var AddDialog, BufferedProcess, CompositeDisposable, CopyDialog, Directory, DirectoryView, Emitter, Minimatch, MoveDialog, RootDragAndDrop, TREE_VIEW_URI, TreeView, _, fs, getFullExtension, getStyleObject, nextId, path, ref, ref1, repoForPath, shell, toggleConfig, updateEditorsForPath,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; },
    slice = [].slice,
    indexOf = [].indexOf || function(item) { for (var i = 0, l = this.length; i < l; i++) { if (i in this && this[i] === item) return i; } return -1; };

  path = require('path');

  shell = require('electron').shell;

  _ = require('underscore-plus');

  ref = require('atom'), BufferedProcess = ref.BufferedProcess, CompositeDisposable = ref.CompositeDisposable, Emitter = ref.Emitter;

  ref1 = require("./helpers"), repoForPath = ref1.repoForPath, getStyleObject = ref1.getStyleObject, getFullExtension = ref1.getFullExtension, updateEditorsForPath = ref1.updateEditorsForPath;

  fs = require('fs-plus');

  AddDialog = require('./add-dialog');

  MoveDialog = require('./move-dialog');

  CopyDialog = require('./copy-dialog');

  Minimatch = null;

  Directory = require('./directory');

  DirectoryView = require('./directory-view');

  RootDragAndDrop = require('./root-drag-and-drop');

  TREE_VIEW_URI = 'atom://tree-view';

  toggleConfig = function(keyPath) {
    return atom.config.set(keyPath, !atom.config.get(keyPath));
  };

  nextId = 1;

  module.exports = TreeView = (function() {
    function TreeView(state) {
      this.onDragLeave = bind(this.onDragLeave, this);
      this.onDragEnter = bind(this.onDragEnter, this);
      this.onStylesheetsChanged = bind(this.onStylesheetsChanged, this);
      var observer;
      this.id = nextId++;
      this.element = document.createElement('div');
      this.element.classList.add('tool-panel', 'tree-view');
      this.element.tabIndex = -1;
      this.list = document.createElement('ol');
      this.list.classList.add('full-menu', 'list-tree', 'has-collapsable-children', 'focusable-panel');
      this.element.appendChild(this.list);
      this.disposables = new CompositeDisposable;
      this.emitter = new Emitter;
      this.roots = [];
      this.selectedPath = null;
      this.ignoredPatterns = [];
      this.useSyncFS = false;
      this.currentlyOpening = new Map;
      this.dragEventCounts = new WeakMap;
      this.rootDragAndDrop = new RootDragAndDrop(this);
      this.handleEvents();
      process.nextTick((function(_this) {
        return function() {
          var onStylesheetsChanged;
          _this.onStylesheetsChanged();
          onStylesheetsChanged = _.debounce(_this.onStylesheetsChanged, 100);
          _this.disposables.add(atom.styles.onDidAddStyleElement(onStylesheetsChanged));
          _this.disposables.add(atom.styles.onDidRemoveStyleElement(onStylesheetsChanged));
          return _this.disposables.add(atom.styles.onDidUpdateStyleElement(onStylesheetsChanged));
        };
      })(this));
      this.updateRoots(state.directoryExpansionStates);
      this.selectEntry(this.roots[0]);
      if (state.selectedPath) {
        this.selectEntryForPath(state.selectedPath);
      }
      if ((state.scrollTop != null) || (state.scrollLeft != null)) {
        observer = new IntersectionObserver((function(_this) {
          return function() {
            if (_this.isVisible()) {
              _this.element.scrollTop = state.scrollTop;
              _this.element.scrollLeft = state.scrollLeft;
              return observer.disconnect();
            }
          };
        })(this));
        observer.observe(this.element);
      }
      if (state.width > 0) {
        this.element.style.width = state.width + "px";
      }
      this.disposables.add(this.onEntryMoved(function(arg) {
        var initialPath, newPath;
        initialPath = arg.initialPath, newPath = arg.newPath;
        return updateEditorsForPath(initialPath, newPath);
      }));
      this.disposables.add(this.onEntryDeleted(function(arg) {
        var editor, j, len, path, ref2, ref3, results;
        path = arg.path;
        ref2 = atom.workspace.getTextEditors();
        results = [];
        for (j = 0, len = ref2.length; j < len; j++) {
          editor = ref2[j];
          if (editor != null ? (ref3 = editor.getPath()) != null ? ref3.startsWith(path) : void 0 : void 0) {
            results.push(editor.destroy());
          } else {
            results.push(void 0);
          }
        }
        return results;
      }));
    }

    TreeView.prototype.serialize = function() {
      var ref2;
      return {
        directoryExpansionStates: new (function(roots) {
          var j, len, root;
          for (j = 0, len = roots.length; j < len; j++) {
            root = roots[j];
            this[root.directory.path] = root.directory.serializeExpansionState();
          }
          return this;
        })(this.roots),
        deserializer: 'TreeView',
        selectedPath: (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0,
        scrollLeft: this.element.scrollLeft,
        scrollTop: this.element.scrollTop,
        width: parseInt(this.element.style.width || 0)
      };
    };

    TreeView.prototype.destroy = function() {
      var j, len, ref2, root;
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        root.directory.destroy();
      }
      this.disposables.dispose();
      this.rootDragAndDrop.dispose();
      return this.emitter.emit('did-destroy');
    };

    TreeView.prototype.onDidDestroy = function(callback) {
      return this.emitter.on('did-destroy', callback);
    };

    TreeView.prototype.getTitle = function() {
      return "Project";
    };

    TreeView.prototype.getURI = function() {
      return TREE_VIEW_URI;
    };

    TreeView.prototype.getPreferredLocation = function() {
      if (atom.config.get('tree-view.showOnRightSide')) {
        return 'right';
      } else {
        return 'left';
      }
    };

    TreeView.prototype.getAllowedLocations = function() {
      return ["left", "right"];
    };

    TreeView.prototype.isPermanentDockItem = function() {
      return true;
    };

    TreeView.prototype.getPreferredWidth = function() {
      var result;
      this.list.style.width = 'min-content';
      result = this.list.offsetWidth;
      this.list.style.width = '';
      return result;
    };

    TreeView.prototype.onDirectoryCreated = function(callback) {
      return this.emitter.on('directory-created', callback);
    };

    TreeView.prototype.onEntryCopied = function(callback) {
      return this.emitter.on('entry-copied', callback);
    };

    TreeView.prototype.onEntryDeleted = function(callback) {
      return this.emitter.on('entry-deleted', callback);
    };

    TreeView.prototype.onEntryMoved = function(callback) {
      return this.emitter.on('entry-moved', callback);
    };

    TreeView.prototype.onFileCreated = function(callback) {
      return this.emitter.on('file-created', callback);
    };

    TreeView.prototype.handleEvents = function() {
      this.element.addEventListener('click', (function(_this) {
        return function(e) {
          if (e.target.classList.contains('entries')) {
            return;
          }
          if (!(e.shiftKey || e.metaKey || e.ctrlKey)) {
            return _this.entryClicked(e);
          }
        };
      })(this));
      this.element.addEventListener('mousedown', (function(_this) {
        return function(e) {
          return _this.onMouseDown(e);
        };
      })(this));
      this.element.addEventListener('dragstart', (function(_this) {
        return function(e) {
          return _this.onDragStart(e);
        };
      })(this));
      this.element.addEventListener('dragenter', (function(_this) {
        return function(e) {
          return _this.onDragEnter(e);
        };
      })(this));
      this.element.addEventListener('dragleave', (function(_this) {
        return function(e) {
          return _this.onDragLeave(e);
        };
      })(this));
      this.element.addEventListener('dragover', (function(_this) {
        return function(e) {
          return _this.onDragOver(e);
        };
      })(this));
      this.element.addEventListener('drop', (function(_this) {
        return function(e) {
          return _this.onDrop(e);
        };
      })(this));
      atom.commands.add(this.element, {
        'core:move-up': this.moveUp.bind(this),
        'core:move-down': this.moveDown.bind(this),
        'core:page-up': (function(_this) {
          return function() {
            return _this.pageUp();
          };
        })(this),
        'core:page-down': (function(_this) {
          return function() {
            return _this.pageDown();
          };
        })(this),
        'core:move-to-top': (function(_this) {
          return function() {
            return _this.scrollToTop();
          };
        })(this),
        'core:move-to-bottom': (function(_this) {
          return function() {
            return _this.scrollToBottom();
          };
        })(this),
        'tree-view:expand-item': (function(_this) {
          return function() {
            return _this.openSelectedEntry({
              pending: true
            }, true);
          };
        })(this),
        'tree-view:recursive-expand-directory': (function(_this) {
          return function() {
            return _this.expandDirectory(true);
          };
        })(this),
        'tree-view:collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory();
          };
        })(this),
        'tree-view:recursive-collapse-directory': (function(_this) {
          return function() {
            return _this.collapseDirectory(true);
          };
        })(this),
        'tree-view:open-selected-entry': (function(_this) {
          return function() {
            return _this.openSelectedEntry();
          };
        })(this),
        'tree-view:open-selected-entry-right': (function(_this) {
          return function() {
            return _this.openSelectedEntryRight();
          };
        })(this),
        'tree-view:open-selected-entry-left': (function(_this) {
          return function() {
            return _this.openSelectedEntryLeft();
          };
        })(this),
        'tree-view:open-selected-entry-up': (function(_this) {
          return function() {
            return _this.openSelectedEntryUp();
          };
        })(this),
        'tree-view:open-selected-entry-down': (function(_this) {
          return function() {
            return _this.openSelectedEntryDown();
          };
        })(this),
        'tree-view:move': (function(_this) {
          return function() {
            return _this.moveSelectedEntry();
          };
        })(this),
        'tree-view:copy': (function(_this) {
          return function() {
            return _this.copySelectedEntries();
          };
        })(this),
        'tree-view:cut': (function(_this) {
          return function() {
            return _this.cutSelectedEntries();
          };
        })(this),
        'tree-view:paste': (function(_this) {
          return function() {
            return _this.pasteEntries();
          };
        })(this),
        'tree-view:copy-full-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(false);
          };
        })(this),
        'tree-view:show-in-file-manager': (function(_this) {
          return function() {
            return _this.showSelectedEntryInFileManager();
          };
        })(this),
        'tree-view:open-in-new-window': (function(_this) {
          return function() {
            return _this.openSelectedEntryInNewWindow();
          };
        })(this),
        'tree-view:copy-project-path': (function(_this) {
          return function() {
            return _this.copySelectedEntryPath(true);
          };
        })(this),
        'tree-view:unfocus': (function(_this) {
          return function() {
            return _this.unfocus();
          };
        })(this),
        'tree-view:toggle-vcs-ignored-files': function() {
          return toggleConfig('tree-view.hideVcsIgnoredFiles');
        },
        'tree-view:toggle-ignored-names': function() {
          return toggleConfig('tree-view.hideIgnoredNames');
        },
        'tree-view:remove-project-folder': (function(_this) {
          return function(e) {
            return _this.removeProjectFolder(e);
          };
        })(this)
      });
      [0, 1, 2, 3, 4, 5, 6, 7, 8].forEach((function(_this) {
        return function(index) {
          return atom.commands.add(_this.element, "tree-view:open-selected-entry-in-pane-" + (index + 1), function() {
            return _this.openSelectedEntryInPane(index);
          });
        };
      })(this));
      this.disposables.add(atom.workspace.getCenter().onDidChangeActivePaneItem((function(_this) {
        return function() {
          _this.selectActiveFile();
          if (atom.config.get('tree-view.autoReveal')) {
            return _this.revealActiveFile(false);
          }
        };
      })(this)));
      this.disposables.add(atom.project.onDidChangePaths((function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideVcsIgnoredFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.hideIgnoredNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('core.ignoredNames', (function(_this) {
        return function() {
          if (atom.config.get('tree-view.hideIgnoredNames')) {
            return _this.updateRoots();
          }
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('tree-view.sortFoldersBeforeFiles', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
      return this.disposables.add(atom.config.onDidChange('tree-view.squashDirectoryNames', (function(_this) {
        return function() {
          return _this.updateRoots();
        };
      })(this)));
    };

    TreeView.prototype.toggle = function() {
      return atom.workspace.toggle(this);
    };

    TreeView.prototype.show = function(focus) {
      return atom.workspace.open(this, {
        searchAllPanes: true,
        activatePane: false,
        activateItem: false
      }).then((function(_this) {
        return function() {
          atom.workspace.paneContainerForURI(_this.getURI()).show();
          if (focus) {
            return _this.focus();
          }
        };
      })(this));
    };

    TreeView.prototype.hide = function() {
      return atom.workspace.hide(this);
    };

    TreeView.prototype.focus = function() {
      return this.element.focus();
    };

    TreeView.prototype.unfocus = function() {
      return atom.workspace.getCenter().activate();
    };

    TreeView.prototype.hasFocus = function() {
      return document.activeElement === this.element;
    };

    TreeView.prototype.toggleFocus = function() {
      if (this.hasFocus()) {
        return this.unfocus();
      } else {
        return this.show(true);
      }
    };

    TreeView.prototype.entryClicked = function(e) {
      var entry, isRecursive;
      if (entry = e.target.closest('.entry')) {
        isRecursive = e.altKey || false;
        this.selectEntry(entry);
        if (entry.classList.contains('directory')) {
          return entry.toggleExpansion(isRecursive);
        } else if (entry.classList.contains('file')) {
          return this.fileViewEntryClicked(e);
        }
      }
    };

    TreeView.prototype.fileViewEntryClicked = function(e) {
      var alwaysOpenExisting, detail, filePath, openPromise, ref2;
      filePath = e.target.closest('.entry').getPath();
      detail = (ref2 = e.detail) != null ? ref2 : 1;
      alwaysOpenExisting = atom.config.get('tree-view.alwaysOpenExisting');
      if (detail === 1) {
        if (atom.config.get('core.allowPendingPaneItems')) {
          openPromise = atom.workspace.open(filePath, {
            pending: true,
            activatePane: false,
            searchAllPanes: alwaysOpenExisting
          });
          this.currentlyOpening.set(filePath, openPromise);
          return openPromise.then((function(_this) {
            return function() {
              return _this.currentlyOpening["delete"](filePath);
            };
          })(this));
        }
      } else if (detail === 2) {
        return this.openAfterPromise(filePath, {
          searchAllPanes: alwaysOpenExisting
        });
      }
    };

    TreeView.prototype.openAfterPromise = function(uri, options) {
      var promise;
      if (promise = this.currentlyOpening.get(uri)) {
        return promise.then(function() {
          return atom.workspace.open(uri, options);
        });
      } else {
        return atom.workspace.open(uri, options);
      }
    };

    TreeView.prototype.loadIgnoredPatterns = function() {
      var error, ignoredName, ignoredNames, j, len, ref2, results;
      this.ignoredPatterns.length = 0;
      if (!atom.config.get('tree-view.hideIgnoredNames')) {
        return;
      }
      if (Minimatch == null) {
        Minimatch = require('minimatch').Minimatch;
      }
      ignoredNames = (ref2 = atom.config.get('core.ignoredNames')) != null ? ref2 : [];
      if (typeof ignoredNames === 'string') {
        ignoredNames = [ignoredNames];
      }
      results = [];
      for (j = 0, len = ignoredNames.length; j < len; j++) {
        ignoredName = ignoredNames[j];
        if (ignoredName) {
          try {
            results.push(this.ignoredPatterns.push(new Minimatch(ignoredName, {
              matchBase: true,
              dot: true
            })));
          } catch (error1) {
            error = error1;
            results.push(atom.notifications.addWarning("Error parsing ignore pattern (" + ignoredName + ")", {
              detail: error.message
            }));
          }
        }
      }
      return results;
    };

    TreeView.prototype.updateRoots = function(expansionStates) {
      var directory, j, key, len, oldExpansionStates, projectPath, ref2, root, stats;
      if (expansionStates == null) {
        expansionStates = {};
      }
      oldExpansionStates = {};
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        oldExpansionStates[root.directory.path] = root.directory.serializeExpansionState();
        root.directory.destroy();
        root.remove();
      }
      this.loadIgnoredPatterns();
      return this.roots = (function() {
        var k, l, len1, len2, ref3, ref4, ref5, ref6, results;
        ref3 = atom.project.getPaths();
        results = [];
        for (k = 0, len1 = ref3.length; k < len1; k++) {
          projectPath = ref3[k];
          stats = fs.lstatSyncNoException(projectPath);
          if (!stats) {
            continue;
          }
          stats = _.pick.apply(_, [stats].concat(slice.call(_.keys(stats))));
          ref4 = ["atime", "birthtime", "ctime", "mtime"];
          for (l = 0, len2 = ref4.length; l < len2; l++) {
            key = ref4[l];
            stats[key] = stats[key].getTime();
          }
          directory = new Directory({
            name: path.basename(projectPath),
            fullPath: projectPath,
            symlink: false,
            isRoot: true,
            expansionState: (ref5 = (ref6 = expansionStates[projectPath]) != null ? ref6 : oldExpansionStates[projectPath]) != null ? ref5 : {
              isExpanded: true
            },
            ignoredPatterns: this.ignoredPatterns,
            useSyncFS: this.useSyncFS,
            stats: stats
          });
          root = new DirectoryView(directory).element;
          this.list.appendChild(root);
          results.push(root);
        }
        return results;
      }).call(this);
    };

    TreeView.prototype.getActivePath = function() {
      var ref2;
      return (ref2 = atom.workspace.getCenter().getActivePaneItem()) != null ? typeof ref2.getPath === "function" ? ref2.getPath() : void 0 : void 0;
    };

    TreeView.prototype.selectActiveFile = function() {
      var activeFilePath;
      if (activeFilePath = this.getActivePath()) {
        return this.selectEntryForPath(activeFilePath);
      }
    };

    TreeView.prototype.revealActiveFile = function(focus) {
      if (_.isEmpty(atom.project.getPaths())) {
        return Promise.resolve();
      }
      return this.show(focus != null ? focus : atom.config.get('tree-view.focusOnReveal')).then((function(_this) {
        return function() {
          var activeFilePath, activePathComponents, currentPath, entry, j, len, pathComponent, ref2, relativePath, results, rootPath;
          if (!(activeFilePath = _this.getActivePath())) {
            return;
          }
          ref2 = atom.project.relativizePath(activeFilePath), rootPath = ref2[0], relativePath = ref2[1];
          if (rootPath == null) {
            return;
          }
          activePathComponents = relativePath.split(path.sep);
          currentPath = rootPath;
          results = [];
          for (j = 0, len = activePathComponents.length; j < len; j++) {
            pathComponent = activePathComponents[j];
            currentPath += path.sep + pathComponent;
            entry = _this.entryForPath(currentPath);
            if (entry.classList.contains('directory')) {
              results.push(entry.expand());
            } else {
              _this.selectEntry(entry);
              results.push(_this.scrollToEntry(entry));
            }
          }
          return results;
        };
      })(this));
    };

    TreeView.prototype.copySelectedEntryPath = function(relativePath) {
      var pathToCopy;
      if (relativePath == null) {
        relativePath = false;
      }
      if (pathToCopy = this.selectedPath) {
        if (relativePath) {
          pathToCopy = atom.project.relativize(pathToCopy);
        }
        return atom.clipboard.write(pathToCopy);
      }
    };

    TreeView.prototype.entryForPath = function(entryPath) {
      var bestMatchEntry, bestMatchLength, entry, entryLength, j, len, ref2, ref3;
      bestMatchEntry = null;
      bestMatchLength = 0;
      ref2 = this.list.querySelectorAll('.entry');
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        if (entry.isPathEqual(entryPath)) {
          return entry;
        }
        entryLength = entry.getPath().length;
        if (((ref3 = entry.directory) != null ? ref3.contains(entryPath) : void 0) && entryLength > bestMatchLength) {
          bestMatchEntry = entry;
          bestMatchLength = entryLength;
        }
      }
      return bestMatchEntry;
    };

    TreeView.prototype.selectEntryForPath = function(entryPath) {
      return this.selectEntry(this.entryForPath(entryPath));
    };

    TreeView.prototype.moveDown = function(event) {
      var nextEntry, selectedEntry;
      if (event != null) {
        event.stopImmediatePropagation();
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (selectedEntry.classList.contains('directory')) {
          if (this.selectEntry(selectedEntry.entries.children[0])) {
            this.scrollToEntry(this.selectedEntry());
            return;
          }
        }
        if (nextEntry = this.nextEntry(selectedEntry)) {
          this.selectEntry(nextEntry);
        }
      } else {
        this.selectEntry(this.roots[0]);
      }
      return this.scrollToEntry(this.selectedEntry());
    };

    TreeView.prototype.moveUp = function(event) {
      var entries, previousEntry, selectedEntry;
      event.stopImmediatePropagation();
      selectedEntry = this.selectedEntry();
      if (selectedEntry != null) {
        if (previousEntry = this.previousEntry(selectedEntry)) {
          this.selectEntry(previousEntry);
          if (previousEntry.classList.contains('directory')) {
            this.selectEntry(_.last(previousEntry.entries.children));
          }
        } else {
          this.selectEntry(selectedEntry.parentElement.closest('.directory'));
        }
      } else {
        entries = this.list.querySelectorAll('.entry');
        this.selectEntry(entries[entries.length - 1]);
      }
      return this.scrollToEntry(this.selectedEntry());
    };

    TreeView.prototype.nextEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        if (currentEntry.nextSibling != null) {
          currentEntry = currentEntry.nextSibling;
          if (currentEntry.matches('.entry')) {
            return currentEntry;
          }
        } else {
          currentEntry = currentEntry.parentElement.closest('.directory');
        }
      }
      return null;
    };

    TreeView.prototype.previousEntry = function(entry) {
      var currentEntry;
      currentEntry = entry;
      while (currentEntry != null) {
        currentEntry = currentEntry.previousSibling;
        if (currentEntry != null ? currentEntry.matches('.entry') : void 0) {
          return currentEntry;
        }
      }
      return null;
    };

    TreeView.prototype.expandDirectory = function(isRecursive) {
      var selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (isRecursive === false && selectedEntry.isExpanded) {
        if (selectedEntry.directory.getEntries().length > 0) {
          return this.moveDown();
        }
      } else {
        return selectedEntry.expand(isRecursive);
      }
    };

    TreeView.prototype.collapseDirectory = function(isRecursive) {
      var directory, selectedEntry;
      if (isRecursive == null) {
        isRecursive = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (directory = selectedEntry.closest('.expanded.directory')) {
        directory.collapse(isRecursive);
        return this.selectEntry(directory);
      }
    };

    TreeView.prototype.openSelectedEntry = function(options, expandDirectory) {
      var selectedEntry;
      if (options == null) {
        options = {};
      }
      if (expandDirectory == null) {
        expandDirectory = false;
      }
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      if (selectedEntry.classList.contains('directory')) {
        if (expandDirectory) {
          return this.expandDirectory(false);
        } else {
          return selectedEntry.toggleExpansion();
        }
      } else if (selectedEntry.classList.contains('file')) {
        if (atom.config.get('tree-view.alwaysOpenExisting')) {
          options = Object.assign({
            searchAllPanes: true
          }, options);
        }
        return this.openAfterPromise(selectedEntry.getPath(), options);
      }
    };

    TreeView.prototype.openSelectedEntrySplit = function(orientation, side) {
      var pane, selectedEntry, split;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getActivePane();
      if (pane && selectedEntry.classList.contains('file')) {
        if (atom.workspace.getCenter().getActivePaneItem()) {
          split = pane.split(orientation, side);
          return atom.workspace.openURIInPane(selectedEntry.getPath(), split);
        } else {
          return this.openSelectedEntry(true);
        }
      }
    };

    TreeView.prototype.openSelectedEntryRight = function() {
      return this.openSelectedEntrySplit('horizontal', 'after');
    };

    TreeView.prototype.openSelectedEntryLeft = function() {
      return this.openSelectedEntrySplit('horizontal', 'before');
    };

    TreeView.prototype.openSelectedEntryUp = function() {
      return this.openSelectedEntrySplit('vertical', 'before');
    };

    TreeView.prototype.openSelectedEntryDown = function() {
      return this.openSelectedEntrySplit('vertical', 'after');
    };

    TreeView.prototype.openSelectedEntryInPane = function(index) {
      var pane, selectedEntry;
      selectedEntry = this.selectedEntry();
      if (selectedEntry == null) {
        return;
      }
      pane = atom.workspace.getCenter().getPanes()[index];
      if (pane && selectedEntry.classList.contains('file')) {
        return atom.workspace.openURIInPane(selectedEntry.getPath(), pane);
      }
    };

    TreeView.prototype.moveSelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if ((entry == null) || indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry.getPath();
      } else {
        oldPath = this.getActivePath();
      }
      if (oldPath) {
        dialog = new MoveDialog(oldPath, {
          onMove: (function(_this) {
            return function(arg) {
              var initialPath, newPath;
              initialPath = arg.initialPath, newPath = arg.newPath;
              return _this.emitter.emit('entry-moved', {
                initialPath: initialPath,
                newPath: newPath
              });
            };
          })(this)
        });
        return dialog.attach();
      }
    };

    TreeView.prototype.fileManagerCommandForPath = function(pathToOpen, isFile) {
      var args, command;
      switch (process.platform) {
        case 'darwin':
          return {
            command: 'open',
            label: 'Finder',
            args: ['-R', pathToOpen]
          };
        case 'win32':
          args = ["/select,\"" + pathToOpen + "\""];
          if (process.env.SystemRoot) {
            command = path.join(process.env.SystemRoot, 'explorer.exe');
          } else {
            command = 'explorer.exe';
          }
          return {
            command: command,
            label: 'Explorer',
            args: args
          };
        default:
          if (isFile) {
            pathToOpen = path.dirname(pathToOpen);
          }
          return {
            command: 'xdg-open',
            label: 'File Manager',
            args: [pathToOpen]
          };
      }
    };

    TreeView.prototype.openInFileManager = function(command, args, label, isFile) {
      var errorLines, exit, handleError, showProcess, stderr;
      handleError = function(errorMessage) {
        return atom.notifications.addError("Opening " + (isFile ? 'file' : 'folder') + " in " + label + " failed", {
          detail: errorMessage,
          dismissable: true
        });
      };
      errorLines = [];
      stderr = function(lines) {
        return errorLines.push(lines);
      };
      exit = function(code) {
        var errorMessage, failed;
        failed = code !== 0;
        errorMessage = errorLines.join('\n');
        if (process.platform === 'win32' && code === 1 && !errorMessage) {
          failed = false;
        }
        if (failed) {
          return handleError(errorMessage);
        }
      };
      showProcess = new BufferedProcess({
        command: command,
        args: args,
        stderr: stderr,
        exit: exit
      });
      showProcess.onWillThrowError(function(arg) {
        var error, handle;
        error = arg.error, handle = arg.handle;
        handle();
        return handleError(error != null ? error.message : void 0);
      });
      return showProcess;
    };

    TreeView.prototype.showSelectedEntryInFileManager = function() {
      var args, command, entry, isFile, label, ref2;
      if (!(entry = this.selectedEntry())) {
        return;
      }
      isFile = entry.classList.contains('file');
      ref2 = this.fileManagerCommandForPath(entry.getPath(), isFile), command = ref2.command, args = ref2.args, label = ref2.label;
      return this.openInFileManager(command, args, label, isFile);
    };

    TreeView.prototype.showCurrentFileInFileManager = function() {
      var args, command, editor, label, ref2;
      if (!(editor = atom.workspace.getCenter().getActiveTextEditor())) {
        return;
      }
      if (!editor.getPath()) {
        return;
      }
      ref2 = this.fileManagerCommandForPath(editor.getPath(), true), command = ref2.command, args = ref2.args, label = ref2.label;
      return this.openInFileManager(command, args, label, true);
    };

    TreeView.prototype.openSelectedEntryInNewWindow = function() {
      var pathToOpen, ref2;
      if (pathToOpen = (ref2 = this.selectedEntry()) != null ? ref2.getPath() : void 0) {
        return atom.open({
          pathsToOpen: [pathToOpen],
          newWindow: true
        });
      }
    };

    TreeView.prototype.copySelectedEntry = function() {
      var dialog, entry, oldPath;
      if (this.hasFocus()) {
        entry = this.selectedEntry();
        if (indexOf.call(this.roots, entry) >= 0) {
          return;
        }
        oldPath = entry != null ? entry.getPath() : void 0;
      } else {
        oldPath = this.getActivePath();
      }
      if (!oldPath) {
        return;
      }
      dialog = new CopyDialog(oldPath, {
        onCopy: (function(_this) {
          return function(arg) {
            var initialPath, newPath;
            initialPath = arg.initialPath, newPath = arg.newPath;
            return _this.emitter.emit('entry-copied', {
              initialPath: initialPath,
              newPath: newPath
            });
          };
        })(this)
      });
      return dialog.attach();
    };

    TreeView.prototype.removeSelectedEntries = function() {
      var activePath, j, len, ref2, ref3, root, selectedPaths;
      if (this.hasFocus()) {
        selectedPaths = this.selectedPaths();
      } else if (activePath = this.getActivePath()) {
        selectedPaths = [activePath];
      }
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      ref2 = this.roots;
      for (j = 0, len = ref2.length; j < len; j++) {
        root = ref2[j];
        if (ref3 = root.getPath(), indexOf.call(selectedPaths, ref3) >= 0) {
          atom.confirm({
            message: "The root directory '" + root.directory.name + "' can't be removed.",
            buttons: ['OK']
          });
          return;
        }
      }
      return atom.confirm({
        message: "Are you sure you want to delete the selected " + (selectedPaths.length > 1 ? 'items' : 'item') + "?",
        detailedMessage: "You are deleting:\n" + (selectedPaths.join('\n')),
        buttons: {
          "Move to Trash": (function(_this) {
            return function() {
              var failedDeletions, k, len1, repo, selectedPath;
              failedDeletions = [];
              for (k = 0, len1 = selectedPaths.length; k < len1; k++) {
                selectedPath = selectedPaths[k];
                if (shell.moveItemToTrash(selectedPath)) {
                  _this.emitter.emit('entry-deleted', {
                    path: selectedPath
                  });
                } else {
                  failedDeletions.push("" + selectedPath);
                }
                if (repo = repoForPath(selectedPath)) {
                  repo.getPathStatus(selectedPath);
                }
              }
              if (failedDeletions.length > 0) {
                atom.notifications.addError(_this.formatTrashFailureMessage(failedDeletions), {
                  description: _this.formatTrashEnabledMessage(),
                  detail: "" + (failedDeletions.join('\n')),
                  dismissable: true
                });
              }
              if (atom.config.get('tree-view.squashDirectoryNames')) {
                return _this.updateRoots();
              }
            };
          })(this),
          "Cancel": null
        }
      });
    };

    TreeView.prototype.formatTrashFailureMessage = function(failedDeletions) {
      var fileText;
      fileText = failedDeletions.length > 1 ? 'files' : 'file';
      return "The following " + fileText + " couldn't be moved to the trash.";
    };

    TreeView.prototype.formatTrashEnabledMessage = function() {
      switch (process.platform) {
        case 'linux':
          return 'Is `gvfs-trash` installed?';
        case 'darwin':
          return 'Is Trash enabled on the volume where the files are stored?';
        case 'win32':
          return 'Is there a Recycle Bin on the drive where the files are stored?';
      }
    };

    TreeView.prototype.copySelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:cutPath');
      return window.localStorage['tree-view:copyPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.cutSelectedEntries = function() {
      var selectedPaths;
      selectedPaths = this.selectedPaths();
      if (!(selectedPaths && selectedPaths.length > 0)) {
        return;
      }
      window.localStorage.removeItem('tree-view:copyPath');
      return window.localStorage['tree-view:cutPath'] = JSON.stringify(selectedPaths);
    };

    TreeView.prototype.pasteEntries = function() {
      var basePath, catchAndShowFileErrors, copiedPaths, cutPaths, extension, fileCounter, filePath, initialPath, initialPathIsDirectory, initialPaths, j, len, newPath, originalNewPath, ref2, results, selectedEntry;
      selectedEntry = this.selectedEntry();
      cutPaths = window.localStorage['tree-view:cutPath'] ? JSON.parse(window.localStorage['tree-view:cutPath']) : null;
      copiedPaths = window.localStorage['tree-view:copyPath'] ? JSON.parse(window.localStorage['tree-view:copyPath']) : null;
      initialPaths = copiedPaths || cutPaths;
      catchAndShowFileErrors = function(operation) {
        var error;
        try {
          return operation();
        } catch (error1) {
          error = error1;
          return atom.notifications.addWarning("Unable to paste paths: " + initialPaths, {
            detail: error.message
          });
        }
      };
      ref2 = initialPaths != null ? initialPaths : [];
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        initialPath = ref2[j];
        initialPathIsDirectory = fs.isDirectorySync(initialPath);
        if (selectedEntry && initialPath && fs.existsSync(initialPath)) {
          basePath = selectedEntry.getPath();
          if (selectedEntry.classList.contains('file')) {
            basePath = path.dirname(basePath);
          }
          newPath = path.join(basePath, path.basename(initialPath));
          if (copiedPaths) {
            fileCounter = 0;
            originalNewPath = newPath;
            while (fs.existsSync(newPath)) {
              if (initialPathIsDirectory) {
                newPath = "" + originalNewPath + fileCounter;
              } else {
                extension = getFullExtension(originalNewPath);
                filePath = path.join(path.dirname(originalNewPath), path.basename(originalNewPath, extension));
                newPath = "" + filePath + fileCounter + extension;
              }
              fileCounter += 1;
            }
            if (fs.isDirectorySync(initialPath)) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.copySync(initialPath, newPath);
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.writeFileSync(newPath, fs.readFileSync(initialPath));
                  return _this.emitter.emit('entry-copied', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            }
          } else if (cutPaths) {
            if (!(fs.existsSync(newPath) || newPath.startsWith(initialPath))) {
              results.push(catchAndShowFileErrors((function(_this) {
                return function() {
                  fs.moveSync(initialPath, newPath);
                  return _this.emitter.emit('entry-moved', {
                    initialPath: initialPath,
                    newPath: newPath
                  });
                };
              })(this)));
            } else {
              results.push(void 0);
            }
          } else {
            results.push(void 0);
          }
        } else {
          results.push(void 0);
        }
      }
      return results;
    };

    TreeView.prototype.add = function(isCreatingFile) {
      var dialog, ref2, ref3, selectedEntry, selectedPath;
      selectedEntry = (ref2 = this.selectedEntry()) != null ? ref2 : this.roots[0];
      selectedPath = (ref3 = selectedEntry != null ? selectedEntry.getPath() : void 0) != null ? ref3 : '';
      dialog = new AddDialog(selectedPath, isCreatingFile);
      dialog.onDidCreateDirectory((function(_this) {
        return function(createdPath) {
          var ref4;
          if ((ref4 = _this.entryForPath(createdPath)) != null) {
            ref4.reload();
          }
          _this.selectEntryForPath(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('directory-created', {
            path: createdPath
          });
        };
      })(this));
      dialog.onDidCreateFile((function(_this) {
        return function(createdPath) {
          atom.workspace.open(createdPath);
          if (atom.config.get('tree-view.squashDirectoryNames')) {
            _this.updateRoots();
          }
          return _this.emitter.emit('file-created', {
            path: createdPath
          });
        };
      })(this));
      return dialog.attach();
    };

    TreeView.prototype.removeProjectFolder = function(e) {
      var pathToRemove, ref2, ref3;
      pathToRemove = (ref2 = e.target.closest(".project-root > .header")) != null ? (ref3 = ref2.querySelector(".name")) != null ? ref3.dataset.path : void 0 : void 0;
      if (atom.project.removePath != null) {
        if (pathToRemove != null) {
          return atom.project.removePath(pathToRemove);
        }
      }
    };

    TreeView.prototype.selectedEntry = function() {
      return this.list.querySelector('.selected');
    };

    TreeView.prototype.selectEntry = function(entry) {
      var selectedEntries;
      if (entry == null) {
        return;
      }
      this.selectedPath = entry.getPath();
      selectedEntries = this.getSelectedEntries();
      if (selectedEntries.length > 1 || selectedEntries[0] !== entry) {
        this.deselect(selectedEntries);
        entry.classList.add('selected');
      }
      return entry;
    };

    TreeView.prototype.getSelectedEntries = function() {
      return this.list.querySelectorAll('.selected');
    };

    TreeView.prototype.deselect = function(elementsToDeselect) {
      var j, len, selected;
      if (elementsToDeselect == null) {
        elementsToDeselect = this.getSelectedEntries();
      }
      for (j = 0, len = elementsToDeselect.length; j < len; j++) {
        selected = elementsToDeselect[j];
        selected.classList.remove('selected');
      }
      return void 0;
    };

    TreeView.prototype.scrollTop = function(top) {
      if (top != null) {
        return this.element.scrollTop = top;
      } else {
        return this.element.scrollTop;
      }
    };

    TreeView.prototype.scrollBottom = function(bottom) {
      if (bottom != null) {
        return this.element.scrollTop = bottom - this.element.offsetHeight;
      } else {
        return this.element.scrollTop + this.element.offsetHeight;
      }
    };

    TreeView.prototype.scrollToEntry = function(entry) {
      var element;
      element = (entry != null ? entry.classList.contains('directory') : void 0) ? entry.header : entry;
      return element != null ? element.scrollIntoViewIfNeeded(true) : void 0;
    };

    TreeView.prototype.scrollToBottom = function() {
      var lastEntry;
      if (lastEntry = _.last(this.list.querySelectorAll('.entry'))) {
        this.selectEntry(lastEntry);
        return this.scrollToEntry(lastEntry);
      }
    };

    TreeView.prototype.scrollToTop = function() {
      if (this.roots[0] != null) {
        this.selectEntry(this.roots[0]);
      }
      return this.element.scrollTop = 0;
    };

    TreeView.prototype.pageUp = function() {
      return this.element.scrollTop -= this.element.offsetHeight;
    };

    TreeView.prototype.pageDown = function() {
      return this.element.scrollTop += this.element.offsetHeight;
    };

    TreeView.prototype.moveEntry = function(initialPath, newDirectoryPath) {
      var entryName, error, newPath, repo;
      if (initialPath === newDirectoryPath) {
        return;
      }
      entryName = path.basename(initialPath);
      newPath = (newDirectoryPath + "/" + entryName).replace(/\s+$/, '');
      try {
        if (!fs.existsSync(newDirectoryPath)) {
          fs.makeTreeSync(newDirectoryPath);
        }
        fs.moveSync(initialPath, newPath);
        this.emitter.emit('entry-moved', {
          initialPath: initialPath,
          newPath: newPath
        });
        if (repo = repoForPath(newPath)) {
          repo.getPathStatus(initialPath);
          return repo.getPathStatus(newPath);
        }
      } catch (error1) {
        error = error1;
        return atom.notifications.addWarning("Failed to move entry " + initialPath + " to " + newDirectoryPath, {
          detail: error.message
        });
      }
    };

    TreeView.prototype.onStylesheetsChanged = function() {
      if (!this.isVisible()) {
        return;
      }
      this.element.style.display = 'none';
      this.element.offsetWidth;
      return this.element.style.display = '';
    };

    TreeView.prototype.onMouseDown = function(e) {
      var entryToSelect;
      if (entryToSelect = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.multiSelectEnabled() && e.target.classList.contains('selected') && (e.button === 2 || e.ctrlKey && process.platform === 'darwin')) {
          return;
        }
        if (e.shiftKey) {
          this.selectContinuousEntries(entryToSelect);
          return this.showMultiSelectMenu();
        } else if (e.metaKey || (e.ctrlKey && process.platform !== 'darwin')) {
          this.selectMultipleEntries(entryToSelect);
          if (this.selectedPaths().length > 1) {
            return this.showMultiSelectMenu();
          }
        } else {
          this.selectEntry(entryToSelect);
          return this.showFullMenu();
        }
      }
    };

    TreeView.prototype.selectedPaths = function() {
      var entry, j, len, ref2, results;
      ref2 = this.getSelectedEntries();
      results = [];
      for (j = 0, len = ref2.length; j < len; j++) {
        entry = ref2[j];
        results.push(entry.getPath());
      }
      return results;
    };

    TreeView.prototype.selectContinuousEntries = function(entry) {
      var currentSelectedEntry, element, elements, entries, entryIndex, i, j, len, parentContainer, selectedIndex;
      currentSelectedEntry = this.selectedEntry();
      parentContainer = entry.parentElement;
      if (parentContainer.contains(currentSelectedEntry)) {
        entries = Array.from(parentContainer.querySelectorAll('.entry'));
        entryIndex = entries.indexOf(entry);
        selectedIndex = entries.indexOf(currentSelectedEntry);
        elements = (function() {
          var j, ref2, ref3, results;
          results = [];
          for (i = j = ref2 = entryIndex, ref3 = selectedIndex; ref2 <= ref3 ? j <= ref3 : j >= ref3; i = ref2 <= ref3 ? ++j : --j) {
            results.push(entries[i]);
          }
          return results;
        })();
        this.deselect();
        for (j = 0, len = elements.length; j < len; j++) {
          element = elements[j];
          element.classList.add('selected');
        }
      }
      return elements;
    };

    TreeView.prototype.selectMultipleEntries = function(entry) {
      if (entry != null) {
        entry.classList.toggle('selected');
      }
      return entry;
    };

    TreeView.prototype.showFullMenu = function() {
      this.list.classList.remove('multi-select');
      return this.list.classList.add('full-menu');
    };

    TreeView.prototype.showMultiSelectMenu = function() {
      this.list.classList.remove('full-menu');
      return this.list.classList.add('multi-select');
    };

    TreeView.prototype.multiSelectEnabled = function() {
      return this.list.classList.contains('multi-select');
    };

    TreeView.prototype.onDragEnter = function(e) {
      var entry, header;
      if (header = e.target.closest('.entry.directory > .header')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        entry = header.parentNode;
        if (!this.dragEventCounts.get(entry)) {
          this.dragEventCounts.set(entry, 0);
        }
        if (this.dragEventCounts.get(entry) === 0) {
          entry.classList.add('selected');
        }
        return this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) + 1);
      }
    };

    TreeView.prototype.onDragLeave = function(e) {
      var entry, header;
      if (header = e.target.closest('.entry.directory > .header')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.stopPropagation();
        entry = header.parentNode;
        this.dragEventCounts.set(entry, this.dragEventCounts.get(entry) - 1);
        if (this.dragEventCounts.get(entry) === 0) {
          return entry.classList.remove('selected');
        }
      }
    };

    TreeView.prototype.onDragStart = function(e) {
      var entry, fileNameElement, initialPath, key, ref2, target, value;
      if (entry = e.target.closest('.entry')) {
        e.stopPropagation();
        if (this.rootDragAndDrop.canDragStart(e)) {
          return this.rootDragAndDrop.onDragStart(e);
        }
        target = entry.querySelector(".name");
        initialPath = target.dataset.path;
        fileNameElement = target.cloneNode(true);
        ref2 = getStyleObject(target);
        for (key in ref2) {
          value = ref2[key];
          fileNameElement.style[key] = value;
        }
        fileNameElement.style.position = 'absolute';
        fileNameElement.style.top = 0;
        fileNameElement.style.left = 0;
        fileNameElement.style.willChange = 'transform';
        document.body.appendChild(fileNameElement);
        e.dataTransfer.effectAllowed = "move";
        e.dataTransfer.setDragImage(fileNameElement, 0, 0);
        e.dataTransfer.setData("initialPath", initialPath);
        return window.requestAnimationFrame(function() {
          return fileNameElement.remove();
        });
      }
    };

    TreeView.prototype.onDragOver = function(e) {
      var entry;
      if (entry = e.target.closest('.entry')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.dragEventCounts.get(entry) > 0 && !entry.classList.contains('selected')) {
          return entry.classList.add('selected');
        }
      }
    };

    TreeView.prototype.onDrop = function(e) {
      var entry, file, initialPath, j, len, newDirectoryPath, ref2, ref3, results;
      if (entry = e.target.closest('.entry')) {
        if (this.rootDragAndDrop.isDragging(e)) {
          return;
        }
        e.preventDefault();
        e.stopPropagation();
        entry.classList.remove('selected');
        if (!entry.classList.contains('directory')) {
          return;
        }
        newDirectoryPath = (ref2 = entry.querySelector('.name')) != null ? ref2.dataset.path : void 0;
        if (!newDirectoryPath) {
          return false;
        }
        initialPath = e.dataTransfer.getData("initialPath");
        if (initialPath) {
          return this.moveEntry(initialPath, newDirectoryPath);
        } else {
          ref3 = e.dataTransfer.files;
          results = [];
          for (j = 0, len = ref3.length; j < len; j++) {
            file = ref3[j];
            results.push(this.moveEntry(file.path, newDirectoryPath));
          }
          return results;
        }
      }
    };

    TreeView.prototype.isVisible = function() {
      return this.element.offsetWidth !== 0 || this.element.offsetHeight !== 0;
    };

    return TreeView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
