(function() {
  var CompositeDisposable, CorrectionsView, SpellCheckTask, SpellCheckView, _,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  CompositeDisposable = require('atom').CompositeDisposable;

  SpellCheckTask = require('./spell-check-task');

  CorrectionsView = null;

  module.exports = SpellCheckView = (function() {
    function SpellCheckView(editor, task, spellCheckModule, getInstance) {
      this.editor = editor;
      this.task = task;
      this.spellCheckModule = spellCheckModule;
      this.getInstance = getInstance;
      this.makeCorrection = bind(this.makeCorrection, this);
      this.addContextMenuEntries = bind(this.addContextMenuEntries, this);
      this.disposables = new CompositeDisposable;
      this.initializeMarkerLayer();
      this.taskWrapper = new SpellCheckTask(this.task);
      this.correctMisspellingCommand = atom.commands.add(atom.views.getView(this.editor), 'spell-check:correct-misspelling', (function(_this) {
        return function() {
          var marker, ref;
          if (marker = _this.markerLayer.findMarkers({
            containsBufferPosition: _this.editor.getCursorBufferPosition()
          })[0]) {
            if (CorrectionsView == null) {
              CorrectionsView = require('./corrections-view');
            }
            if ((ref = _this.correctionsView) != null) {
              ref.destroy();
            }
            _this.correctionsView = new CorrectionsView(_this.editor, _this.getCorrections(marker), marker, _this, _this.updateMisspellings);
            return _this.correctionsView.attach();
          }
        };
      })(this));
      atom.views.getView(this.editor).addEventListener('contextmenu', this.addContextMenuEntries);
      this.taskWrapper.onDidSpellCheck((function(_this) {
        return function(misspellings) {
          _this.destroyMarkers();
          if (_this.buffer != null) {
            return _this.addMarkers(misspellings);
          }
        };
      })(this));
      this.disposables.add(this.editor.onDidChangePath((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(this.editor.onDidChangeGrammar((function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('editor.fontSize', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.disposables.add(atom.config.onDidChange('spell-check.grammars', (function(_this) {
        return function() {
          return _this.subscribeToBuffer();
        };
      })(this)));
      this.subscribeToBuffer();
      this.disposables.add(this.editor.onDidDestroy(this.destroy.bind(this)));
    }

    SpellCheckView.prototype.initializeMarkerLayer = function() {
      this.markerLayer = this.editor.addMarkerLayer({
        maintainHistory: false
      });
      return this.markerLayerDecoration = this.editor.decorateMarkerLayer(this.markerLayer, {
        type: 'highlight',
        "class": 'spell-check-misspelling',
        deprecatedRegionClass: 'misspelling'
      });
    };

    SpellCheckView.prototype.destroy = function() {
      var ref;
      this.unsubscribeFromBuffer();
      this.disposables.dispose();
      this.taskWrapper.terminate();
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      this.correctMisspellingCommand.dispose();
      if ((ref = this.correctionsView) != null) {
        ref.destroy();
      }
      return this.clearContextMenuEntries();
    };

    SpellCheckView.prototype.unsubscribeFromBuffer = function() {
      this.destroyMarkers();
      if (this.buffer != null) {
        this.bufferDisposable.dispose();
        return this.buffer = null;
      }
    };

    SpellCheckView.prototype.subscribeToBuffer = function() {
      this.unsubscribeFromBuffer();
      if (this.spellCheckCurrentGrammar()) {
        this.buffer = this.editor.getBuffer();
        this.bufferDisposable = this.buffer.onDidStopChanging((function(_this) {
          return function() {
            return _this.updateMisspellings();
          };
        })(this));
        return this.updateMisspellings();
      }
    };

    SpellCheckView.prototype.spellCheckCurrentGrammar = function() {
      var grammar;
      grammar = this.editor.getGrammar().scopeName;
      return _.contains(atom.config.get('spell-check.grammars'), grammar);
    };

    SpellCheckView.prototype.destroyMarkers = function() {
      this.markerLayer.destroy();
      this.markerLayerDecoration.destroy();
      return this.initializeMarkerLayer();
    };

    SpellCheckView.prototype.addMarkers = function(misspellings) {
      var i, len, misspelling, results;
      results = [];
      for (i = 0, len = misspellings.length; i < len; i++) {
        misspelling = misspellings[i];
        results.push(this.markerLayer.markBufferRange(misspelling, {
          invalidate: 'touch'
        }));
      }
      return results;
    };

    SpellCheckView.prototype.updateMisspellings = function() {
      var error, ref;
      try {
        return this.taskWrapper.start(this.editor.buffer);
      } catch (error1) {
        error = error1;
        return console.warn('Error starting spell check task', (ref = error.stack) != null ? ref : error);
      }
    };

    SpellCheckView.prototype.getCorrections = function(marker) {
      var args, instance, misspelling, projectPath, ref, ref1, ref2, relativePath;
      projectPath = null;
      relativePath = null;
      if ((ref = this.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
        ref2 = atom.project.relativizePath(this.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
      }
      args = {
        projectPath: projectPath,
        relativePath: relativePath
      };
      instance = this.getInstance();
      misspelling = this.editor.getTextInBufferRange(marker.getBufferRange());
      return instance.suggest(args, misspelling);
    };

    SpellCheckView.prototype.addContextMenuEntries = function(mouseEvent) {
      var commandName, contextMenuEntry, correction, correctionIndex, corrections, currentBufferPosition, currentScreenPosition, i, len, marker;
      this.clearContextMenuEntries();
      currentScreenPosition = atom.views.getView(this.editor).component.screenPositionForMouseEvent(mouseEvent);
      currentBufferPosition = this.editor.bufferPositionForScreenPosition(currentScreenPosition);
      if (marker = this.markerLayer.findMarkers({
        containsBufferPosition: currentBufferPosition
      })[0]) {
        corrections = this.getCorrections(marker);
        if (corrections.length > 0) {
          this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
          correctionIndex = 0;
          for (i = 0, len = corrections.length; i < len; i++) {
            correction = corrections[i];
            contextMenuEntry = {};
            commandName = 'spell-check:correct-misspelling-' + correctionIndex;
            contextMenuEntry.command = (function(_this) {
              return function(correction, contextMenuEntry) {
                return atom.commands.add(atom.views.getView(_this.editor), commandName, function() {
                  _this.makeCorrection(correction, marker);
                  return _this.clearContextMenuEntries();
                });
              };
            })(this)(correction, contextMenuEntry);
            contextMenuEntry.menuItem = atom.contextMenu.add({
              'atom-text-editor': [
                {
                  label: correction.label,
                  command: commandName
                }
              ]
            });
            this.spellCheckModule.contextMenuEntries.push(contextMenuEntry);
            correctionIndex++;
          }
          return this.spellCheckModule.contextMenuEntries.push({
            menuItem: atom.contextMenu.add({
              'atom-text-editor': [
                {
                  type: 'separator'
                }
              ]
            })
          });
        }
      }
    };

    SpellCheckView.prototype.makeCorrection = function(correction, marker) {
      var args, projectPath, ref, ref1, ref2, relativePath;
      if (correction.isSuggestion) {
        this.editor.setSelectedBufferRange(marker.getBufferRange());
        return this.editor.insertText(correction.suggestion);
      } else {
        projectPath = null;
        relativePath = null;
        if ((ref = this.editor.buffer) != null ? (ref1 = ref.file) != null ? ref1.path : void 0 : void 0) {
          ref2 = atom.project.relativizePath(this.editor.buffer.file.path), projectPath = ref2[0], relativePath = ref2[1];
        }
        args = {
          id: this.id,
          projectPath: projectPath,
          relativePath: relativePath
        };
        correction.plugin.add(args, correction);
        return this.updateMisspellings.bind(this)();
      }
    };

    SpellCheckView.prototype.clearContextMenuEntries = function() {
      var entry, i, len, ref, ref1, ref2;
      ref = this.spellCheckModule.contextMenuEntries;
      for (i = 0, len = ref.length; i < len; i++) {
        entry = ref[i];
        if ((ref1 = entry.command) != null) {
          ref1.dispose();
        }
        if ((ref2 = entry.menuItem) != null) {
          ref2.dispose();
        }
      }
      return this.spellCheckModule.contextMenuEntries = [];
    };

    return SpellCheckView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiL2hvbWUvYW5kcmVpL2F0b20tMS4xOS4yL291dC9hcHAvbm9kZV9tb2R1bGVzL3NwZWxsLWNoZWNrL2xpYi9zcGVsbC1jaGVjay12aWV3LmNvZmZlZSJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQTtBQUFBLE1BQUEsdUVBQUE7SUFBQTs7RUFBQSxDQUFBLEdBQUksT0FBQSxDQUFRLGlCQUFSOztFQUNILHNCQUF1QixPQUFBLENBQVEsTUFBUjs7RUFDeEIsY0FBQSxHQUFpQixPQUFBLENBQVEsb0JBQVI7O0VBRWpCLGVBQUEsR0FBa0I7O0VBRWxCLE1BQU0sQ0FBQyxPQUFQLEdBQ007SUFDUyx3QkFBQyxNQUFELEVBQVUsSUFBVixFQUFpQixnQkFBakIsRUFBb0MsV0FBcEM7TUFBQyxJQUFDLENBQUEsU0FBRDtNQUFTLElBQUMsQ0FBQSxPQUFEO01BQU8sSUFBQyxDQUFBLG1CQUFEO01BQW1CLElBQUMsQ0FBQSxjQUFEOzs7TUFDL0MsSUFBQyxDQUFBLFdBQUQsR0FBZSxJQUFJO01BQ25CLElBQUMsQ0FBQSxxQkFBRCxDQUFBO01BQ0EsSUFBQyxDQUFBLFdBQUQsR0FBbUIsSUFBQSxjQUFBLENBQWUsSUFBQyxDQUFBLElBQWhCO01BRW5CLElBQUMsQ0FBQSx5QkFBRCxHQUE2QixJQUFJLENBQUMsUUFBUSxDQUFDLEdBQWQsQ0FBa0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFYLENBQW1CLElBQUMsQ0FBQSxNQUFwQixDQUFsQixFQUErQyxpQ0FBL0MsRUFBa0YsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFBO0FBQzdHLGNBQUE7VUFBQSxJQUFHLE1BQUEsR0FBUyxLQUFDLENBQUEsV0FBVyxDQUFDLFdBQWIsQ0FBeUI7WUFBQyxzQkFBQSxFQUF3QixLQUFDLENBQUEsTUFBTSxDQUFDLHVCQUFSLENBQUEsQ0FBekI7V0FBekIsQ0FBc0YsQ0FBQSxDQUFBLENBQWxHOztjQUNFLGtCQUFtQixPQUFBLENBQVEsb0JBQVI7OztpQkFDSCxDQUFFLE9BQWxCLENBQUE7O1lBQ0EsS0FBQyxDQUFBLGVBQUQsR0FBdUIsSUFBQSxlQUFBLENBQWdCLEtBQUMsQ0FBQSxNQUFqQixFQUF5QixLQUFDLENBQUEsY0FBRCxDQUFnQixNQUFoQixDQUF6QixFQUFrRCxNQUFsRCxFQUEwRCxLQUExRCxFQUFnRSxLQUFDLENBQUEsa0JBQWpFO21CQUN2QixLQUFDLENBQUEsZUFBZSxDQUFDLE1BQWpCLENBQUEsRUFKRjs7UUFENkc7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWxGO01BTzdCLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBWCxDQUFtQixJQUFDLENBQUEsTUFBcEIsQ0FBMkIsQ0FBQyxnQkFBNUIsQ0FBNkMsYUFBN0MsRUFBNEQsSUFBQyxDQUFBLHFCQUE3RDtNQUVBLElBQUMsQ0FBQSxXQUFXLENBQUMsZUFBYixDQUE2QixDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUMsWUFBRDtVQUMzQixLQUFDLENBQUEsY0FBRCxDQUFBO1VBQ0EsSUFBNkIsb0JBQTdCO21CQUFBLEtBQUMsQ0FBQSxVQUFELENBQVksWUFBWixFQUFBOztRQUYyQjtNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBN0I7TUFJQSxJQUFDLENBQUEsV0FBVyxDQUFDLEdBQWIsQ0FBaUIsSUFBQyxDQUFBLE1BQU0sQ0FBQyxlQUFSLENBQXdCLENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQTtpQkFDdkMsS0FBQyxDQUFBLGlCQUFELENBQUE7UUFEdUM7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQXhCLENBQWpCO01BR0EsSUFBQyxDQUFBLFdBQVcsQ0FBQyxHQUFiLENBQWlCLElBQUMsQ0FBQSxNQUFNLENBQUMsa0JBQVIsQ0FBMkIsQ0FBQSxTQUFBLEtBQUE7ZUFBQSxTQUFBO2lCQUMxQyxLQUFDLENBQUEsaUJBQUQsQ0FBQTtRQUQwQztNQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBM0IsQ0FBakI7TUFHQSxJQUFDLENBQUEsV0FBVyxDQUFDLEdBQWIsQ0FBaUIsSUFBSSxDQUFDLE1BQU0sQ0FBQyxXQUFaLENBQXdCLGlCQUF4QixFQUEyQyxDQUFBLFNBQUEsS0FBQTtlQUFBLFNBQUE7aUJBQzFELEtBQUMsQ0FBQSxpQkFBRCxDQUFBO1FBRDBEO01BQUEsQ0FBQSxDQUFBLENBQUEsSUFBQSxDQUEzQyxDQUFqQjtNQUdBLElBQUMsQ0FBQSxXQUFXLENBQUMsR0FBYixDQUFpQixJQUFJLENBQUMsTUFBTSxDQUFDLFdBQVosQ0FBd0Isc0JBQXhCLEVBQWdELENBQUEsU0FBQSxLQUFBO2VBQUEsU0FBQTtpQkFDL0QsS0FBQyxDQUFBLGlCQUFELENBQUE7UUFEK0Q7TUFBQSxDQUFBLENBQUEsQ0FBQSxJQUFBLENBQWhELENBQWpCO01BR0EsSUFBQyxDQUFBLGlCQUFELENBQUE7TUFFQSxJQUFDLENBQUEsV0FBVyxDQUFDLEdBQWIsQ0FBaUIsSUFBQyxDQUFBLE1BQU0sQ0FBQyxZQUFSLENBQXFCLElBQUMsQ0FBQSxPQUFPLENBQUMsSUFBVCxDQUFjLElBQWQsQ0FBckIsQ0FBakI7SUFoQ1c7OzZCQWtDYixxQkFBQSxHQUF1QixTQUFBO01BQ3JCLElBQUMsQ0FBQSxXQUFELEdBQWUsSUFBQyxDQUFBLE1BQU0sQ0FBQyxjQUFSLENBQXVCO1FBQUMsZUFBQSxFQUFpQixLQUFsQjtPQUF2QjthQUNmLElBQUMsQ0FBQSxxQkFBRCxHQUF5QixJQUFDLENBQUEsTUFBTSxDQUFDLG1CQUFSLENBQTRCLElBQUMsQ0FBQSxXQUE3QixFQUEwQztRQUNqRSxJQUFBLEVBQU0sV0FEMkQ7UUFFakUsQ0FBQSxLQUFBLENBQUEsRUFBTyx5QkFGMEQ7UUFHakUscUJBQUEsRUFBdUIsYUFIMEM7T0FBMUM7SUFGSjs7NkJBUXZCLE9BQUEsR0FBUyxTQUFBO0FBQ1AsVUFBQTtNQUFBLElBQUMsQ0FBQSxxQkFBRCxDQUFBO01BQ0EsSUFBQyxDQUFBLFdBQVcsQ0FBQyxPQUFiLENBQUE7TUFDQSxJQUFDLENBQUEsV0FBVyxDQUFDLFNBQWIsQ0FBQTtNQUNBLElBQUMsQ0FBQSxXQUFXLENBQUMsT0FBYixDQUFBO01BQ0EsSUFBQyxDQUFBLHFCQUFxQixDQUFDLE9BQXZCLENBQUE7TUFDQSxJQUFDLENBQUEseUJBQXlCLENBQUMsT0FBM0IsQ0FBQTs7V0FDZ0IsQ0FBRSxPQUFsQixDQUFBOzthQUNBLElBQUMsQ0FBQSx1QkFBRCxDQUFBO0lBUk87OzZCQVVULHFCQUFBLEdBQXVCLFNBQUE7TUFDckIsSUFBQyxDQUFBLGNBQUQsQ0FBQTtNQUVBLElBQUcsbUJBQUg7UUFDRSxJQUFDLENBQUEsZ0JBQWdCLENBQUMsT0FBbEIsQ0FBQTtlQUNBLElBQUMsQ0FBQSxNQUFELEdBQVUsS0FGWjs7SUFIcUI7OzZCQU92QixpQkFBQSxHQUFtQixTQUFBO01BQ2pCLElBQUMsQ0FBQSxxQkFBRCxDQUFBO01BRUEsSUFBRyxJQUFDLENBQUEsd0JBQUQsQ0FBQSxDQUFIO1FBQ0UsSUFBQyxDQUFBLE1BQUQsR0FBVSxJQUFDLENBQUEsTUFBTSxDQUFDLFNBQVIsQ0FBQTtRQUNWLElBQUMsQ0FBQSxnQkFBRCxHQUFvQixJQUFDLENBQUEsTUFBTSxDQUFDLGlCQUFSLENBQTBCLENBQUEsU0FBQSxLQUFBO2lCQUFBLFNBQUE7bUJBQUcsS0FBQyxDQUFBLGtCQUFELENBQUE7VUFBSDtRQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBMUI7ZUFDcEIsSUFBQyxDQUFBLGtCQUFELENBQUEsRUFIRjs7SUFIaUI7OzZCQVFuQix3QkFBQSxHQUEwQixTQUFBO0FBQ3hCLFVBQUE7TUFBQSxPQUFBLEdBQVUsSUFBQyxDQUFBLE1BQU0sQ0FBQyxVQUFSLENBQUEsQ0FBb0IsQ0FBQzthQUMvQixDQUFDLENBQUMsUUFBRixDQUFXLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBWixDQUFnQixzQkFBaEIsQ0FBWCxFQUFvRCxPQUFwRDtJQUZ3Qjs7NkJBSTFCLGNBQUEsR0FBZ0IsU0FBQTtNQUNkLElBQUMsQ0FBQSxXQUFXLENBQUMsT0FBYixDQUFBO01BQ0EsSUFBQyxDQUFBLHFCQUFxQixDQUFDLE9BQXZCLENBQUE7YUFDQSxJQUFDLENBQUEscUJBQUQsQ0FBQTtJQUhjOzs2QkFLaEIsVUFBQSxHQUFZLFNBQUMsWUFBRDtBQUNWLFVBQUE7QUFBQTtXQUFBLDhDQUFBOztxQkFDRSxJQUFDLENBQUEsV0FBVyxDQUFDLGVBQWIsQ0FBNkIsV0FBN0IsRUFBMEM7VUFBQyxVQUFBLEVBQVksT0FBYjtTQUExQztBQURGOztJQURVOzs2QkFJWixrQkFBQSxHQUFvQixTQUFBO0FBRWxCLFVBQUE7QUFBQTtlQUNFLElBQUMsQ0FBQSxXQUFXLENBQUMsS0FBYixDQUFtQixJQUFDLENBQUEsTUFBTSxDQUFDLE1BQTNCLEVBREY7T0FBQSxjQUFBO1FBRU07ZUFDSixPQUFPLENBQUMsSUFBUixDQUFhLGlDQUFiLHNDQUE4RCxLQUE5RCxFQUhGOztJQUZrQjs7NkJBT3BCLGNBQUEsR0FBZ0IsU0FBQyxNQUFEO0FBRWQsVUFBQTtNQUFBLFdBQUEsR0FBYztNQUNkLFlBQUEsR0FBZTtNQUNmLGtFQUFnQixDQUFFLHNCQUFsQjtRQUNFLE9BQThCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYixDQUE0QixJQUFDLENBQUEsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUF6QyxDQUE5QixFQUFDLHFCQUFELEVBQWMsdUJBRGhCOztNQUVBLElBQUEsR0FBTztRQUNMLFdBQUEsRUFBYSxXQURSO1FBRUwsWUFBQSxFQUFjLFlBRlQ7O01BTVAsUUFBQSxHQUFXLElBQUMsQ0FBQSxXQUFELENBQUE7TUFDWCxXQUFBLEdBQWMsSUFBQyxDQUFBLE1BQU0sQ0FBQyxvQkFBUixDQUE2QixNQUFNLENBQUMsY0FBUCxDQUFBLENBQTdCO2FBQ2QsUUFBUSxDQUFDLE9BQVQsQ0FBaUIsSUFBakIsRUFBdUIsV0FBdkI7SUFkYzs7NkJBZ0JoQixxQkFBQSxHQUF1QixTQUFDLFVBQUQ7QUFDckIsVUFBQTtNQUFBLElBQUMsQ0FBQSx1QkFBRCxDQUFBO01BSUEscUJBQUEsR0FBd0IsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFYLENBQW1CLElBQUMsQ0FBQSxNQUFwQixDQUEyQixDQUFDLFNBQVMsQ0FBQywyQkFBdEMsQ0FBa0UsVUFBbEU7TUFDeEIscUJBQUEsR0FBd0IsSUFBQyxDQUFBLE1BQU0sQ0FBQywrQkFBUixDQUF3QyxxQkFBeEM7TUFHeEIsSUFBRyxNQUFBLEdBQVMsSUFBQyxDQUFBLFdBQVcsQ0FBQyxXQUFiLENBQXlCO1FBQUMsc0JBQUEsRUFBd0IscUJBQXpCO09BQXpCLENBQTBFLENBQUEsQ0FBQSxDQUF0RjtRQUNFLFdBQUEsR0FBYyxJQUFDLENBQUEsY0FBRCxDQUFnQixNQUFoQjtRQUNkLElBQUcsV0FBVyxDQUFDLE1BQVosR0FBcUIsQ0FBeEI7VUFDRSxJQUFDLENBQUEsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUMsSUFBckMsQ0FBMEM7WUFDeEMsUUFBQSxFQUFVLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBakIsQ0FBcUI7Y0FBQyxrQkFBQSxFQUFvQjtnQkFBQztrQkFBQyxJQUFBLEVBQU0sV0FBUDtpQkFBRDtlQUFyQjthQUFyQixDQUQ4QjtXQUExQztVQUlBLGVBQUEsR0FBa0I7QUFDbEIsZUFBQSw2Q0FBQTs7WUFDRSxnQkFBQSxHQUFtQjtZQUVuQixXQUFBLEdBQWMsa0NBQUEsR0FBcUM7WUFDbkQsZ0JBQWdCLENBQUMsT0FBakIsR0FBOEIsQ0FBQSxTQUFBLEtBQUE7cUJBQUEsU0FBQyxVQUFELEVBQWEsZ0JBQWI7dUJBQzVCLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBZCxDQUFrQixJQUFJLENBQUMsS0FBSyxDQUFDLE9BQVgsQ0FBbUIsS0FBQyxDQUFBLE1BQXBCLENBQWxCLEVBQStDLFdBQS9DLEVBQTRELFNBQUE7a0JBQzFELEtBQUMsQ0FBQSxjQUFELENBQWdCLFVBQWhCLEVBQTRCLE1BQTVCO3lCQUNBLEtBQUMsQ0FBQSx1QkFBRCxDQUFBO2dCQUYwRCxDQUE1RDtjQUQ0QjtZQUFBLENBQUEsQ0FBQSxDQUFBLElBQUEsQ0FBSCxDQUFJLFVBQUosRUFBZ0IsZ0JBQWhCO1lBTTNCLGdCQUFnQixDQUFDLFFBQWpCLEdBQTRCLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBakIsQ0FBcUI7Y0FDL0Msa0JBQUEsRUFBb0I7Z0JBQUM7a0JBQUMsS0FBQSxFQUFPLFVBQVUsQ0FBQyxLQUFuQjtrQkFBMEIsT0FBQSxFQUFTLFdBQW5DO2lCQUFEO2VBRDJCO2FBQXJCO1lBRzVCLElBQUMsQ0FBQSxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFyQyxDQUEwQyxnQkFBMUM7WUFDQSxlQUFBO0FBZEY7aUJBZ0JBLElBQUMsQ0FBQSxnQkFBZ0IsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFyQyxDQUEwQztZQUN4QyxRQUFBLEVBQVUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFqQixDQUFxQjtjQUFDLGtCQUFBLEVBQW9CO2dCQUFDO2tCQUFDLElBQUEsRUFBTSxXQUFQO2lCQUFEO2VBQXJCO2FBQXJCLENBRDhCO1dBQTFDLEVBdEJGO1NBRkY7O0lBVHFCOzs2QkFxQ3ZCLGNBQUEsR0FBZ0IsU0FBQyxVQUFELEVBQWEsTUFBYjtBQUNkLFVBQUE7TUFBQSxJQUFHLFVBQVUsQ0FBQyxZQUFkO1FBRUUsSUFBQyxDQUFBLE1BQU0sQ0FBQyxzQkFBUixDQUErQixNQUFNLENBQUMsY0FBUCxDQUFBLENBQS9CO2VBQ0EsSUFBQyxDQUFBLE1BQU0sQ0FBQyxVQUFSLENBQW1CLFVBQVUsQ0FBQyxVQUE5QixFQUhGO09BQUEsTUFBQTtRQU1FLFdBQUEsR0FBYztRQUNkLFlBQUEsR0FBZTtRQUNmLHlFQUF1QixDQUFFLHNCQUF6QjtVQUNFLE9BQThCLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYixDQUE0QixJQUFDLENBQUEsTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBaEQsQ0FBOUIsRUFBQyxxQkFBRCxFQUFjLHVCQURoQjs7UUFFQSxJQUFBLEdBQU87VUFDTCxFQUFBLEVBQUksSUFBQyxDQUFBLEVBREE7VUFFTCxXQUFBLEVBQWEsV0FGUjtVQUdMLFlBQUEsRUFBYyxZQUhUOztRQU9QLFVBQVUsQ0FBQyxNQUFNLENBQUMsR0FBbEIsQ0FBc0IsSUFBdEIsRUFBNEIsVUFBNUI7ZUFHQSxJQUFDLENBQUEsa0JBQWtCLENBQUMsSUFBcEIsQ0FBeUIsSUFBekIsQ0FBQSxDQUFBLEVBcEJGOztJQURjOzs2QkF1QmhCLHVCQUFBLEdBQXlCLFNBQUE7QUFDdkIsVUFBQTtBQUFBO0FBQUEsV0FBQSxxQ0FBQTs7O2NBQ2UsQ0FBRSxPQUFmLENBQUE7OztjQUNjLENBQUUsT0FBaEIsQ0FBQTs7QUFGRjthQUlBLElBQUMsQ0FBQSxnQkFBZ0IsQ0FBQyxrQkFBbEIsR0FBdUM7SUFMaEI7Ozs7O0FBM0szQiIsInNvdXJjZXNDb250ZW50IjpbIl8gPSByZXF1aXJlICd1bmRlcnNjb3JlLXBsdXMnXG57Q29tcG9zaXRlRGlzcG9zYWJsZX0gPSByZXF1aXJlICdhdG9tJ1xuU3BlbGxDaGVja1Rhc2sgPSByZXF1aXJlICcuL3NwZWxsLWNoZWNrLXRhc2snXG5cbkNvcnJlY3Rpb25zVmlldyA9IG51bGxcblxubW9kdWxlLmV4cG9ydHMgPVxuY2xhc3MgU3BlbGxDaGVja1ZpZXdcbiAgY29uc3RydWN0b3I6IChAZWRpdG9yLCBAdGFzaywgQHNwZWxsQ2hlY2tNb2R1bGUsIEBnZXRJbnN0YW5jZSkgLT5cbiAgICBAZGlzcG9zYWJsZXMgPSBuZXcgQ29tcG9zaXRlRGlzcG9zYWJsZVxuICAgIEBpbml0aWFsaXplTWFya2VyTGF5ZXIoKVxuICAgIEB0YXNrV3JhcHBlciA9IG5ldyBTcGVsbENoZWNrVGFzayBAdGFza1xuXG4gICAgQGNvcnJlY3RNaXNzcGVsbGluZ0NvbW1hbmQgPSBhdG9tLmNvbW1hbmRzLmFkZCBhdG9tLnZpZXdzLmdldFZpZXcoQGVkaXRvciksICdzcGVsbC1jaGVjazpjb3JyZWN0LW1pc3NwZWxsaW5nJywgPT5cbiAgICAgIGlmIG1hcmtlciA9IEBtYXJrZXJMYXllci5maW5kTWFya2Vycyh7Y29udGFpbnNCdWZmZXJQb3NpdGlvbjogQGVkaXRvci5nZXRDdXJzb3JCdWZmZXJQb3NpdGlvbigpfSlbMF1cbiAgICAgICAgQ29ycmVjdGlvbnNWaWV3ID89IHJlcXVpcmUgJy4vY29ycmVjdGlvbnMtdmlldydcbiAgICAgICAgQGNvcnJlY3Rpb25zVmlldz8uZGVzdHJveSgpXG4gICAgICAgIEBjb3JyZWN0aW9uc1ZpZXcgPSBuZXcgQ29ycmVjdGlvbnNWaWV3KEBlZGl0b3IsIEBnZXRDb3JyZWN0aW9ucyhtYXJrZXIpLCBtYXJrZXIsIHRoaXMsIEB1cGRhdGVNaXNzcGVsbGluZ3MpXG4gICAgICAgIEBjb3JyZWN0aW9uc1ZpZXcuYXR0YWNoKClcblxuICAgIGF0b20udmlld3MuZ2V0VmlldyhAZWRpdG9yKS5hZGRFdmVudExpc3RlbmVyICdjb250ZXh0bWVudScsIEBhZGRDb250ZXh0TWVudUVudHJpZXNcblxuICAgIEB0YXNrV3JhcHBlci5vbkRpZFNwZWxsQ2hlY2sgKG1pc3NwZWxsaW5ncykgPT5cbiAgICAgIEBkZXN0cm95TWFya2VycygpXG4gICAgICBAYWRkTWFya2VycyhtaXNzcGVsbGluZ3MpIGlmIEBidWZmZXI/XG5cbiAgICBAZGlzcG9zYWJsZXMuYWRkIEBlZGl0b3Iub25EaWRDaGFuZ2VQYXRoID0+XG4gICAgICBAc3Vic2NyaWJlVG9CdWZmZXIoKVxuXG4gICAgQGRpc3Bvc2FibGVzLmFkZCBAZWRpdG9yLm9uRGlkQ2hhbmdlR3JhbW1hciA9PlxuICAgICAgQHN1YnNjcmliZVRvQnVmZmVyKClcblxuICAgIEBkaXNwb3NhYmxlcy5hZGQgYXRvbS5jb25maWcub25EaWRDaGFuZ2UgJ2VkaXRvci5mb250U2l6ZScsID0+XG4gICAgICBAc3Vic2NyaWJlVG9CdWZmZXIoKVxuXG4gICAgQGRpc3Bvc2FibGVzLmFkZCBhdG9tLmNvbmZpZy5vbkRpZENoYW5nZSAnc3BlbGwtY2hlY2suZ3JhbW1hcnMnLCA9PlxuICAgICAgQHN1YnNjcmliZVRvQnVmZmVyKClcblxuICAgIEBzdWJzY3JpYmVUb0J1ZmZlcigpXG5cbiAgICBAZGlzcG9zYWJsZXMuYWRkIEBlZGl0b3Iub25EaWREZXN0cm95KEBkZXN0cm95LmJpbmQodGhpcykpXG5cbiAgaW5pdGlhbGl6ZU1hcmtlckxheWVyOiAtPlxuICAgIEBtYXJrZXJMYXllciA9IEBlZGl0b3IuYWRkTWFya2VyTGF5ZXIoe21haW50YWluSGlzdG9yeTogZmFsc2V9KVxuICAgIEBtYXJrZXJMYXllckRlY29yYXRpb24gPSBAZWRpdG9yLmRlY29yYXRlTWFya2VyTGF5ZXIoQG1hcmtlckxheWVyLCB7XG4gICAgICB0eXBlOiAnaGlnaGxpZ2h0JyxcbiAgICAgIGNsYXNzOiAnc3BlbGwtY2hlY2stbWlzc3BlbGxpbmcnLFxuICAgICAgZGVwcmVjYXRlZFJlZ2lvbkNsYXNzOiAnbWlzc3BlbGxpbmcnXG4gICAgfSlcblxuICBkZXN0cm95OiAtPlxuICAgIEB1bnN1YnNjcmliZUZyb21CdWZmZXIoKVxuICAgIEBkaXNwb3NhYmxlcy5kaXNwb3NlKClcbiAgICBAdGFza1dyYXBwZXIudGVybWluYXRlKClcbiAgICBAbWFya2VyTGF5ZXIuZGVzdHJveSgpXG4gICAgQG1hcmtlckxheWVyRGVjb3JhdGlvbi5kZXN0cm95KClcbiAgICBAY29ycmVjdE1pc3NwZWxsaW5nQ29tbWFuZC5kaXNwb3NlKClcbiAgICBAY29ycmVjdGlvbnNWaWV3Py5kZXN0cm95KClcbiAgICBAY2xlYXJDb250ZXh0TWVudUVudHJpZXMoKVxuXG4gIHVuc3Vic2NyaWJlRnJvbUJ1ZmZlcjogLT5cbiAgICBAZGVzdHJveU1hcmtlcnMoKVxuXG4gICAgaWYgQGJ1ZmZlcj9cbiAgICAgIEBidWZmZXJEaXNwb3NhYmxlLmRpc3Bvc2UoKVxuICAgICAgQGJ1ZmZlciA9IG51bGxcblxuICBzdWJzY3JpYmVUb0J1ZmZlcjogLT5cbiAgICBAdW5zdWJzY3JpYmVGcm9tQnVmZmVyKClcblxuICAgIGlmIEBzcGVsbENoZWNrQ3VycmVudEdyYW1tYXIoKVxuICAgICAgQGJ1ZmZlciA9IEBlZGl0b3IuZ2V0QnVmZmVyKClcbiAgICAgIEBidWZmZXJEaXNwb3NhYmxlID0gQGJ1ZmZlci5vbkRpZFN0b3BDaGFuZ2luZyA9PiBAdXBkYXRlTWlzc3BlbGxpbmdzKClcbiAgICAgIEB1cGRhdGVNaXNzcGVsbGluZ3MoKVxuXG4gIHNwZWxsQ2hlY2tDdXJyZW50R3JhbW1hcjogLT5cbiAgICBncmFtbWFyID0gQGVkaXRvci5nZXRHcmFtbWFyKCkuc2NvcGVOYW1lXG4gICAgXy5jb250YWlucyhhdG9tLmNvbmZpZy5nZXQoJ3NwZWxsLWNoZWNrLmdyYW1tYXJzJyksIGdyYW1tYXIpXG5cbiAgZGVzdHJveU1hcmtlcnM6IC0+XG4gICAgQG1hcmtlckxheWVyLmRlc3Ryb3koKVxuICAgIEBtYXJrZXJMYXllckRlY29yYXRpb24uZGVzdHJveSgpXG4gICAgQGluaXRpYWxpemVNYXJrZXJMYXllcigpXG5cbiAgYWRkTWFya2VyczogKG1pc3NwZWxsaW5ncykgLT5cbiAgICBmb3IgbWlzc3BlbGxpbmcgaW4gbWlzc3BlbGxpbmdzXG4gICAgICBAbWFya2VyTGF5ZXIubWFya0J1ZmZlclJhbmdlKG1pc3NwZWxsaW5nLCB7aW52YWxpZGF0ZTogJ3RvdWNoJ30pXG5cbiAgdXBkYXRlTWlzc3BlbGxpbmdzOiAtPlxuICAgICMgVGFzazo6c3RhcnQgY2FuIHRocm93IGVycm9ycyBhdG9tL2F0b20jMzMyNlxuICAgIHRyeVxuICAgICAgQHRhc2tXcmFwcGVyLnN0YXJ0IEBlZGl0b3IuYnVmZmVyXG4gICAgY2F0Y2ggZXJyb3JcbiAgICAgIGNvbnNvbGUud2FybignRXJyb3Igc3RhcnRpbmcgc3BlbGwgY2hlY2sgdGFzaycsIGVycm9yLnN0YWNrID8gZXJyb3IpXG5cbiAgZ2V0Q29ycmVjdGlvbnM6IChtYXJrZXIpIC0+XG4gICAgIyBCdWlsZCB1cCB0aGUgYXJndW1lbnRzIG9iamVjdCBmb3IgdGhpcyBidWZmZXIgYW5kIHRleHQuXG4gICAgcHJvamVjdFBhdGggPSBudWxsXG4gICAgcmVsYXRpdmVQYXRoID0gbnVsbFxuICAgIGlmIEBidWZmZXI/LmZpbGU/LnBhdGhcbiAgICAgIFtwcm9qZWN0UGF0aCwgcmVsYXRpdmVQYXRoXSA9IGF0b20ucHJvamVjdC5yZWxhdGl2aXplUGF0aChAYnVmZmVyLmZpbGUucGF0aClcbiAgICBhcmdzID0ge1xuICAgICAgcHJvamVjdFBhdGg6IHByb2plY3RQYXRoLFxuICAgICAgcmVsYXRpdmVQYXRoOiByZWxhdGl2ZVBhdGhcbiAgICB9XG5cbiAgICAjIEdldCB0aGUgbWlzc3BlbGxlZCB3b3JkIGFuZCB0aGVuIHJlcXVlc3QgY29ycmVjdGlvbnMuXG4gICAgaW5zdGFuY2UgPSBAZ2V0SW5zdGFuY2UoKVxuICAgIG1pc3NwZWxsaW5nID0gQGVkaXRvci5nZXRUZXh0SW5CdWZmZXJSYW5nZSBtYXJrZXIuZ2V0QnVmZmVyUmFuZ2UoKVxuICAgIGluc3RhbmNlLnN1Z2dlc3QgYXJncywgbWlzc3BlbGxpbmdcblxuICBhZGRDb250ZXh0TWVudUVudHJpZXM6IChtb3VzZUV2ZW50KSA9PlxuICAgIEBjbGVhckNvbnRleHRNZW51RW50cmllcygpXG4gICAgIyBHZXQgYnVmZmVyIHBvc2l0aW9uIG9mIHRoZSByaWdodCBjbGljayBldmVudC4gSWYgdGhlIGNsaWNrIGhhcHBlbnMgb3V0c2lkZVxuICAgICMgdGhlIGJvdW5kYXJpZXMgb2YgYW55IHRleHQsIHRoZSBtZXRob2QgZGVmYXVsdHMgdG8gdGhlIGJ1ZmZlciBwb3NpdGlvbiBvZlxuICAgICMgdGhlIGxhc3QgY2hhcmFjdGVyIGluIHRoZSBlZGl0b3IuXG4gICAgY3VycmVudFNjcmVlblBvc2l0aW9uID0gYXRvbS52aWV3cy5nZXRWaWV3KEBlZGl0b3IpLmNvbXBvbmVudC5zY3JlZW5Qb3NpdGlvbkZvck1vdXNlRXZlbnQgbW91c2VFdmVudFxuICAgIGN1cnJlbnRCdWZmZXJQb3NpdGlvbiA9IEBlZGl0b3IuYnVmZmVyUG9zaXRpb25Gb3JTY3JlZW5Qb3NpdGlvbihjdXJyZW50U2NyZWVuUG9zaXRpb24pXG5cbiAgICAjIENoZWNrIHRvIHNlZSBpZiB0aGUgc2VsZWN0ZWQgd29yZCBpcyBpbmNvcnJlY3QuXG4gICAgaWYgbWFya2VyID0gQG1hcmtlckxheWVyLmZpbmRNYXJrZXJzKHtjb250YWluc0J1ZmZlclBvc2l0aW9uOiBjdXJyZW50QnVmZmVyUG9zaXRpb259KVswXVxuICAgICAgY29ycmVjdGlvbnMgPSBAZ2V0Q29ycmVjdGlvbnMobWFya2VyKVxuICAgICAgaWYgY29ycmVjdGlvbnMubGVuZ3RoID4gMFxuICAgICAgICBAc3BlbGxDaGVja01vZHVsZS5jb250ZXh0TWVudUVudHJpZXMucHVzaCh7XG4gICAgICAgICAgbWVudUl0ZW06IGF0b20uY29udGV4dE1lbnUuYWRkKHsnYXRvbS10ZXh0LWVkaXRvcic6IFt7dHlwZTogJ3NlcGFyYXRvcid9XX0pXG4gICAgICAgIH0pXG5cbiAgICAgICAgY29ycmVjdGlvbkluZGV4ID0gMFxuICAgICAgICBmb3IgY29ycmVjdGlvbiBpbiBjb3JyZWN0aW9uc1xuICAgICAgICAgIGNvbnRleHRNZW51RW50cnkgPSB7fVxuICAgICAgICAgICMgUmVnaXN0ZXIgbmV3IGNvbW1hbmQgZm9yIGNvcnJlY3Rpb24uXG4gICAgICAgICAgY29tbWFuZE5hbWUgPSAnc3BlbGwtY2hlY2s6Y29ycmVjdC1taXNzcGVsbGluZy0nICsgY29ycmVjdGlvbkluZGV4XG4gICAgICAgICAgY29udGV4dE1lbnVFbnRyeS5jb21tYW5kID0gZG8gKGNvcnJlY3Rpb24sIGNvbnRleHRNZW51RW50cnkpID0+XG4gICAgICAgICAgICBhdG9tLmNvbW1hbmRzLmFkZCBhdG9tLnZpZXdzLmdldFZpZXcoQGVkaXRvciksIGNvbW1hbmROYW1lLCA9PlxuICAgICAgICAgICAgICBAbWFrZUNvcnJlY3Rpb24oY29ycmVjdGlvbiwgbWFya2VyKVxuICAgICAgICAgICAgICBAY2xlYXJDb250ZXh0TWVudUVudHJpZXMoKVxuXG4gICAgICAgICAgIyBBZGQgbmV3IG1lbnUgaXRlbSBmb3IgY29ycmVjdGlvbi5cbiAgICAgICAgICBjb250ZXh0TWVudUVudHJ5Lm1lbnVJdGVtID0gYXRvbS5jb250ZXh0TWVudS5hZGQoe1xuICAgICAgICAgICAgJ2F0b20tdGV4dC1lZGl0b3InOiBbe2xhYmVsOiBjb3JyZWN0aW9uLmxhYmVsLCBjb21tYW5kOiBjb21tYW5kTmFtZX1dXG4gICAgICAgICAgfSlcbiAgICAgICAgICBAc3BlbGxDaGVja01vZHVsZS5jb250ZXh0TWVudUVudHJpZXMucHVzaCBjb250ZXh0TWVudUVudHJ5XG4gICAgICAgICAgY29ycmVjdGlvbkluZGV4KytcblxuICAgICAgICBAc3BlbGxDaGVja01vZHVsZS5jb250ZXh0TWVudUVudHJpZXMucHVzaCh7XG4gICAgICAgICAgbWVudUl0ZW06IGF0b20uY29udGV4dE1lbnUuYWRkKHsnYXRvbS10ZXh0LWVkaXRvcic6IFt7dHlwZTogJ3NlcGFyYXRvcid9XX0pXG4gICAgICAgIH0pXG5cbiAgbWFrZUNvcnJlY3Rpb246IChjb3JyZWN0aW9uLCBtYXJrZXIpID0+XG4gICAgaWYgY29ycmVjdGlvbi5pc1N1Z2dlc3Rpb25cbiAgICAgICMgVXBkYXRlIHRoZSBidWZmZXIgd2l0aCB0aGUgY29ycmVjdGlvbi5cbiAgICAgIEBlZGl0b3Iuc2V0U2VsZWN0ZWRCdWZmZXJSYW5nZShtYXJrZXIuZ2V0QnVmZmVyUmFuZ2UoKSlcbiAgICAgIEBlZGl0b3IuaW5zZXJ0VGV4dChjb3JyZWN0aW9uLnN1Z2dlc3Rpb24pXG4gICAgZWxzZVxuICAgICAgIyBCdWlsZCB1cCB0aGUgYXJndW1lbnRzIG9iamVjdCBmb3IgdGhpcyBidWZmZXIgYW5kIHRleHQuXG4gICAgICBwcm9qZWN0UGF0aCA9IG51bGxcbiAgICAgIHJlbGF0aXZlUGF0aCA9IG51bGxcbiAgICAgIGlmIEBlZGl0b3IuYnVmZmVyPy5maWxlPy5wYXRoXG4gICAgICAgIFtwcm9qZWN0UGF0aCwgcmVsYXRpdmVQYXRoXSA9IGF0b20ucHJvamVjdC5yZWxhdGl2aXplUGF0aChAZWRpdG9yLmJ1ZmZlci5maWxlLnBhdGgpXG4gICAgICBhcmdzID0ge1xuICAgICAgICBpZDogQGlkLFxuICAgICAgICBwcm9qZWN0UGF0aDogcHJvamVjdFBhdGgsXG4gICAgICAgIHJlbGF0aXZlUGF0aDogcmVsYXRpdmVQYXRoXG4gICAgICB9XG5cbiAgICAgICMgU2VuZCB0aGUgXCJhZGRcIiByZXF1ZXN0IHRvIHRoZSBwbHVnaW4uXG4gICAgICBjb3JyZWN0aW9uLnBsdWdpbi5hZGQgYXJncywgY29ycmVjdGlvblxuXG4gICAgICAjIFVwZGF0ZSB0aGUgYnVmZmVyIHRvIGhhbmRsZSB0aGUgY29ycmVjdGlvbnMuXG4gICAgICBAdXBkYXRlTWlzc3BlbGxpbmdzLmJpbmQodGhpcykoKVxuXG4gIGNsZWFyQ29udGV4dE1lbnVFbnRyaWVzOiAtPlxuICAgIGZvciBlbnRyeSBpbiBAc3BlbGxDaGVja01vZHVsZS5jb250ZXh0TWVudUVudHJpZXNcbiAgICAgIGVudHJ5LmNvbW1hbmQ/LmRpc3Bvc2UoKVxuICAgICAgZW50cnkubWVudUl0ZW0/LmRpc3Bvc2UoKVxuXG4gICAgQHNwZWxsQ2hlY2tNb2R1bGUuY29udGV4dE1lbnVFbnRyaWVzID0gW11cbiJdfQ==
