(function() {
  var SpellCheckerManager, manager;

  SpellCheckerManager = (function() {
    function SpellCheckerManager() {}

    SpellCheckerManager.prototype.checkers = [];

    SpellCheckerManager.prototype.checkerPaths = [];

    SpellCheckerManager.prototype.locales = [];

    SpellCheckerManager.prototype.localePaths = [];

    SpellCheckerManager.prototype.useLocales = false;

    SpellCheckerManager.prototype.localeCheckers = null;

    SpellCheckerManager.prototype.knownWords = [];

    SpellCheckerManager.prototype.addKnownWords = false;

    SpellCheckerManager.prototype.knownWordsChecker = null;

    SpellCheckerManager.prototype.isTask = false;

    SpellCheckerManager.prototype.setGlobalArgs = function(data) {
      var _, changed, checker, checkers, i, len, ref, removeKnownWordsChecker, removeLocaleCheckers;
      _ = require("underscore-plus");
      changed = false;
      removeLocaleCheckers = false;
      removeKnownWordsChecker = false;
      if (!_.isEqual(this.locales, data.locales)) {
        if (!this.localeCheckers || ((ref = data.locales) != null ? ref.length : void 0) !== 0) {
          this.locales = data.locales;
          removeLocaleCheckers = true;
        }
      }
      if (!_.isEqual(this.localePaths, data.localePaths)) {
        this.localePaths = data.localePaths;
        removeLocaleCheckers = true;
      }
      if (this.useLocales !== data.useLocales) {
        this.useLocales = data.useLocales;
        removeLocaleCheckers = true;
      }
      if (this.knownWords !== data.knownWords) {
        this.knownWords = data.knownWords;
        removeKnownWordsChecker = true;
        changed = true;
      }
      if (this.addKnownWords !== data.addKnownWords) {
        this.addKnownWords = data.addKnownWords;
        removeKnownWordsChecker = true;
      }
      if (removeLocaleCheckers && this.localeCheckers) {
        checkers = this.localeCheckers;
        for (i = 0, len = checkers.length; i < len; i++) {
          checker = checkers[i];
          this.removeSpellChecker(checker);
        }
        this.localeCheckers = null;
        changed = true;
      }
      if (removeKnownWordsChecker && this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        this.knownWordsChecker = null;
        changed = true;
      }
      if (changed) {
        return this.emitSettingsChanged();
      }
    };

    SpellCheckerManager.prototype.emitSettingsChanged = function() {
      if (this.isTask) {
        return emit("spell-check:settings-changed");
      }
    };

    SpellCheckerManager.prototype.addCheckerPath = function(checkerPath) {
      var checker;
      checker = require(checkerPath);
      return this.addPluginChecker(checker);
    };

    SpellCheckerManager.prototype.addPluginChecker = function(checker) {
      this.addSpellChecker(checker);
      return this.emitSettingsChanged();
    };

    SpellCheckerManager.prototype.addSpellChecker = function(checker) {
      return this.checkers.push(checker);
    };

    SpellCheckerManager.prototype.removeSpellChecker = function(spellChecker) {
      return this.checkers = this.checkers.filter(function(plugin) {
        return plugin !== spellChecker;
      });
    };

    SpellCheckerManager.prototype.check = function(args, text) {
      var checker, correct, i, incorrect, incorrects, intersection, invertedCorrect, j, k, l, len, len1, len2, len3, len4, lineBeginIndex, lineEndIndex, lineRange, m, misspellings, multirange, newIncorrect, range, rangeIndex, rangeRange, ref, ref1, ref2, ref3, removeRange, results, row;
      this.init();
      multirange = require('multi-integer-range');
      correct = new multirange.MultiRange([]);
      incorrects = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSpelling(args)) {
          continue;
        }
        results = checker.check(args, text);
        if (results.invertIncorrectAsCorrect && results.incorrect) {
          invertedCorrect = new multirange.MultiRange([[0, text.length]]);
          removeRange = new multirange.MultiRange([]);
          ref1 = results.incorrect;
          for (j = 0, len1 = ref1.length; j < len1; j++) {
            range = ref1[j];
            removeRange.appendRange(range.start, range.end);
          }
          invertedCorrect.subtract(removeRange);
          correct.append(invertedCorrect);
        } else if (results.correct) {
          ref2 = results.correct;
          for (k = 0, len2 = ref2.length; k < len2; k++) {
            range = ref2[k];
            correct.appendRange(range.start, range.end);
          }
        }
        if (results.incorrect) {
          newIncorrect = new multirange.MultiRange([]);
          incorrects.push(newIncorrect);
          ref3 = results.incorrect;
          for (l = 0, len3 = ref3.length; l < len3; l++) {
            range = ref3[l];
            newIncorrect.appendRange(range.start, range.end);
          }
        }
      }
      if (incorrects.length === 0) {
        return {
          id: args.id,
          misspellings: []
        };
      }
      intersection = null;
      for (m = 0, len4 = incorrects.length; m < len4; m++) {
        incorrect = incorrects[m];
        if (intersection === null) {
          intersection = incorrect;
        } else {
          intersection.append(incorrect);
        }
      }
      if (intersection.length === 0) {
        return {
          id: args.id,
          misspellings: []
        };
      }
      if (correct.ranges.length > 0) {
        intersection.subtract(correct);
      }
      row = 0;
      rangeIndex = 0;
      lineBeginIndex = 0;
      misspellings = [];
      while (lineBeginIndex < text.length && rangeIndex < intersection.ranges.length) {
        lineEndIndex = text.indexOf('\n', lineBeginIndex);
        if (lineEndIndex === -1) {
          lineEndIndex = 2e308;
        }
        while (true) {
          range = intersection.ranges[rangeIndex];
          if (range && range[0] < lineEndIndex) {
            lineRange = new multirange.MultiRange([]).appendRange(lineBeginIndex, lineEndIndex);
            rangeRange = new multirange.MultiRange([]).appendRange(range[0], range[1]);
            lineRange.intersect(rangeRange);
            this.addMisspellings(misspellings, row, lineRange.ranges[0], lineBeginIndex, text);
            if (lineEndIndex >= range[1]) {
              rangeIndex++;
            } else {
              break;
            }
          } else {
            break;
          }
        }
        lineBeginIndex = lineEndIndex + 1;
        row++;
      }
      return {
        id: args.id,
        misspellings: misspellings
      };
    };

    SpellCheckerManager.prototype.suggest = function(args, word) {
      var checker, i, index, j, k, key, keys, l, len, len1, len2, len3, len4, m, priority, ref, ref1, results, s, seen, suggestion, suggestions, target, targets, that;
      this.init();
      suggestions = [];
      ref = this.checkers;
      for (i = 0, len = ref.length; i < len; i++) {
        checker = ref[i];
        if (!checker.isEnabled() || !checker.providesSuggestions(args)) {
          continue;
        }
        index = 0;
        priority = checker.getPriority();
        ref1 = checker.suggest(args, word);
        for (j = 0, len1 = ref1.length; j < len1; j++) {
          suggestion = ref1[j];
          suggestions.push({
            isSuggestion: true,
            priority: priority,
            index: index++,
            suggestion: suggestion,
            label: suggestion
          });
        }
      }
      keys = Object.keys(suggestions).sort(function(key1, key2) {
        var value1, value2, weight1, weight2;
        value1 = suggestions[key1];
        value2 = suggestions[key2];
        weight1 = value1.priority + value1.index;
        weight2 = value2.priority + value2.index;
        if (weight1 !== weight2) {
          return weight1 - weight2;
        }
        return value1.suggestion.localeCompare(value2.suggestion);
      });
      results = [];
      seen = [];
      for (k = 0, len2 = keys.length; k < len2; k++) {
        key = keys[k];
        s = suggestions[key];
        if (seen.hasOwnProperty(s.suggestion)) {
          continue;
        }
        results.push(s);
        seen[s.suggestion] = 1;
      }
      that = this;
      keys = Object.keys(this.checkers).sort(function(key1, key2) {
        var value1, value2;
        value1 = that.checkers[key1];
        value2 = that.checkers[key2];
        return value1.getPriority() - value2.getPriority();
      });
      for (l = 0, len3 = keys.length; l < len3; l++) {
        key = keys[l];
        checker = this.checkers[key];
        if (!checker.isEnabled() || !checker.providesAdding(args)) {
          continue;
        }
        targets = checker.getAddingTargets(args);
        for (m = 0, len4 = targets.length; m < len4; m++) {
          target = targets[m];
          target.plugin = checker;
          target.word = word;
          target.isSuggestion = false;
          results.push(target);
        }
      }
      return results;
    };

    SpellCheckerManager.prototype.addMisspellings = function(misspellings, row, range, lineBeginIndex, text) {
      var i, len, markBeginIndex, markEndIndex, part, parts, substring, substringIndex;
      substring = text.substring(range[0], range[1]);
      if (/\s+/.test(substring)) {
        parts = substring.split(/(\s+)/);
        substringIndex = 0;
        for (i = 0, len = parts.length; i < len; i++) {
          part = parts[i];
          if (!/\s+/.test(part)) {
            markBeginIndex = range[0] - lineBeginIndex + substringIndex;
            markEndIndex = markBeginIndex + part.length;
            misspellings.push([[row, markBeginIndex], [row, markEndIndex]]);
          }
          substringIndex += part.length;
        }
        return;
      }
      return misspellings.push([[row, range[0] - lineBeginIndex], [row, range[1] - lineBeginIndex]]);
    };

    SpellCheckerManager.prototype.init = function() {
      var KnownWordsChecker, SystemChecker, checker, defaultLocale, i, len, locale, ref, separatorChar;
      if (this.localeCheckers === null) {
        this.localeCheckers = [];
        if (this.useLocales) {
          if (!this.locales.length) {
            defaultLocale = process.env.LANG;
            if (defaultLocale) {
              this.locales = [defaultLocale.split('.')[0]];
            }
          }
          if (!this.locales.length) {
            defaultLocale = navigator.language;
            if (defaultLocale && defaultLocale.length === 5) {
              separatorChar = defaultLocale.charAt(2);
              if (separatorChar === '_' || separatorChar === '-') {
                this.locales = [defaultLocale];
              }
            }
          }
          if (!this.locales.length) {
            this.locales = ['en_US'];
          }
          SystemChecker = require("./system-checker");
          ref = this.locales;
          for (i = 0, len = ref.length; i < len; i++) {
            locale = ref[i];
            checker = new SystemChecker(locale, this.localePaths);
            this.addSpellChecker(checker);
            this.localeCheckers.push(checker);
          }
        }
      }
      if (this.knownWordsChecker === null) {
        KnownWordsChecker = require('./known-words-checker');
        this.knownWordsChecker = new KnownWordsChecker(this.knownWords);
        this.knownWordsChecker.enableAdd = this.addKnownWords;
        return this.addSpellChecker(this.knownWordsChecker);
      }
    };

    SpellCheckerManager.prototype.deactivate = function() {
      this.checkers = [];
      this.locales = [];
      this.localePaths = [];
      this.useLocales = false;
      this.localeCheckers = null;
      this.knownWords = [];
      this.addKnownWords = false;
      return this.knownWordsChecker = null;
    };

    SpellCheckerManager.prototype.reloadLocales = function() {
      var i, len, localeChecker, ref;
      if (this.localeCheckers) {
        ref = this.localeCheckers;
        for (i = 0, len = ref.length; i < len; i++) {
          localeChecker = ref[i];
          this.removeSpellChecker(localeChecker);
        }
        return this.localeCheckers = null;
      }
    };

    SpellCheckerManager.prototype.reloadKnownWords = function() {
      if (this.knownWordsChecker) {
        this.removeSpellChecker(this.knownWordsChecker);
        return this.knownWordsChecker = null;
      }
    };

    return SpellCheckerManager;

  })();

  manager = new SpellCheckerManager;

  module.exports = manager;

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
