(function() {
  var CommandLogger, FileURLRegExp, NotificationIssue, StackTraceParser, TITLE_CHAR_LIMIT, UserUtilities, fs, path,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  fs = require('fs-plus');

  path = require('path');

  StackTraceParser = require('stacktrace-parser');

  CommandLogger = require('./command-logger');

  UserUtilities = require('./user-utilities');

  TITLE_CHAR_LIMIT = 100;

  FileURLRegExp = new RegExp('file://\w*/(.*)');

  module.exports = NotificationIssue = (function() {
    function NotificationIssue(notification) {
      this.notification = notification;
      this.normalizedStackPaths = bind(this.normalizedStackPaths, this);
    }

    NotificationIssue.prototype.findSimilarIssues = function() {
      var githubHeaders, issueTitle, query, repo, repoUrl;
      repoUrl = this.getRepoUrl();
      if (repoUrl == null) {
        repoUrl = 'atom/atom';
      }
      repo = repoUrl.replace(/http(s)?:\/\/(\d+\.)?github.com\//gi, '');
      issueTitle = this.getIssueTitle();
      query = issueTitle + " repo:" + repo;
      githubHeaders = new Headers({
        accept: 'application/vnd.github.v3+json',
        contentType: "application/json"
      });
      return fetch("https://api.github.com/search/issues?q=" + (encodeURIComponent(query)) + "&sort=created", {
        headers: githubHeaders
      }).then(function(r) {
        return r != null ? r.json() : void 0;
      }).then(function(data) {
        var issue, issues, j, len, ref;
        if ((data != null ? data.items : void 0) != null) {
          issues = {};
          ref = data.items;
          for (j = 0, len = ref.length; j < len; j++) {
            issue = ref[j];
            if (issue.title.indexOf(issueTitle) > -1 && (issues[issue.state] == null)) {
              issues[issue.state] = issue;
              if ((issues.open != null) && (issues.closed != null)) {
                return issues;
              }
            }
          }
          if ((issues.open != null) || (issues.closed != null)) {
            return issues;
          }
        }
        return null;
      })["catch"](function(e) {
        return null;
      });
    };

    NotificationIssue.prototype.getIssueUrlForSystem = function() {
      return this.getIssueUrl().then(function(issueUrl) {
        return fetch("https://is.gd/create.php?format=simple", {
          method: 'POST',
          headers: {
            'Content-Type': 'application/x-www-form-urlencoded'
          },
          body: "url=" + (encodeURIComponent(issueUrl))
        }).then(function(r) {
          return r.text();
        })["catch"](function(e) {
          return null;
        });
      });
    };

    NotificationIssue.prototype.getIssueUrl = function() {
      return this.getIssueBody().then((function(_this) {
        return function(issueBody) {
          var repoUrl;
          repoUrl = _this.getRepoUrl();
          if (repoUrl == null) {
            repoUrl = 'https://github.com/atom/atom';
          }
          return repoUrl + "/issues/new?title=" + (_this.encodeURI(_this.getIssueTitle())) + "&body=" + (_this.encodeURI(issueBody));
        };
      })(this));
    };

    NotificationIssue.prototype.encodeURI = function(str) {
      return encodeURI(str).replace(/#/g, '%23').replace(/;/g, '%3B').replace(/%20/g, '+');
    };

    NotificationIssue.prototype.getIssueTitle = function() {
      var title;
      title = this.notification.getMessage();
      title = title.replace(process.env.ATOM_HOME, '$ATOM_HOME');
      if (process.platform === 'win32') {
        title = title.replace(process.env.USERPROFILE, '~');
        title = title.replace(path.sep, path.posix.sep);
      } else {
        title = title.replace(process.env.HOME, '~');
      }
      if (title.length > TITLE_CHAR_LIMIT) {
        title = title.substring(0, TITLE_CHAR_LIMIT - 3) + '...';
      }
      return title.replace(/\r?\n|\r/g, "");
    };

    NotificationIssue.prototype.getIssueBody = function() {
      return new Promise((function(_this) {
        return function(resolve, reject) {
          var nonCorePackagesPromise, systemPromise;
          if (_this.issueBody) {
            return resolve(_this.issueBody);
          }
          systemPromise = UserUtilities.getOSVersion();
          nonCorePackagesPromise = UserUtilities.getNonCorePackages();
          return Promise.all([systemPromise, nonCorePackagesPromise]).then(function(all) {
            var copyText, message, nonCorePackages, options, packageMessage, packageName, packageVersion, ref, ref1, repoUrl, rootUserStatus, systemName, systemUser;
            systemName = all[0], nonCorePackages = all[1];
            message = _this.notification.getMessage();
            options = _this.notification.getOptions();
            repoUrl = _this.getRepoUrl();
            packageName = _this.getPackageName();
            if (packageName != null) {
              packageVersion = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.version : void 0 : void 0;
            }
            copyText = '';
            systemUser = process.env.USER;
            rootUserStatus = '';
            if (systemUser === 'root') {
              rootUserStatus = '**User**: root';
            }
            if ((packageName != null) && (repoUrl != null)) {
              packageMessage = "[" + packageName + "](" + repoUrl + ") package " + packageVersion;
            } else if (packageName != null) {
              packageMessage = "'" + packageName + "' package v" + packageVersion;
            } else {
              packageMessage = 'Atom Core';
            }
            _this.issueBody = "[Enter steps to reproduce:]\n\n1. ...\n2. ...\n\n**Atom**: " + (atom.getVersion()) + " " + process.arch + "\n**Electron**: " + process.versions.electron + "\n**OS**: " + systemName + "\n**Thrown From**: " + packageMessage + "\n" + rootUserStatus + "\n\n### Stack Trace\n\n" + message + "\n\n```\nAt " + options.detail + "\n\n" + (_this.normalizedStackPaths(options.stack)) + "\n```\n\n### Commands\n\n" + (CommandLogger.instance().getText()) + "\n\n### Non-Core Packages\n\n```\n" + (nonCorePackages.join('\n')) + "\n```\n\n" + copyText;
            return resolve(_this.issueBody);
          });
        };
      })(this));
    };

    NotificationIssue.prototype.normalizedStackPaths = function(stack) {
      return stack.replace(/(^\W+at )([\w.]{2,} [(])?(.*)(:\d+:\d+[)]?)/gm, (function(_this) {
        return function(m, p1, p2, p3, p4) {
          return p1 + (p2 || '') + _this.normalizePath(p3) + p4;
        };
      })(this));
    };

    NotificationIssue.prototype.normalizePath = function(path) {
      return path.replace('file:///', '').replace(/[\/]/g, '\\').replace(fs.getHomeDirectory(), '~').replace(/\\/g, '/').replace(/.*(\/(app\.asar|packages\/).*)/, '$1');
    };

    NotificationIssue.prototype.getRepoUrl = function() {
      var packageName, packagePath, ref, ref1, ref2, ref3, ref4, repo, repoUrl;
      packageName = this.getPackageName();
      if (packageName == null) {
        return;
      }
      repo = (ref = atom.packages.getLoadedPackage(packageName)) != null ? (ref1 = ref.metadata) != null ? ref1.repository : void 0 : void 0;
      repoUrl = (ref2 = repo != null ? repo.url : void 0) != null ? ref2 : repo;
      if (!repoUrl) {
        if (packagePath = atom.packages.resolvePackagePath(packageName)) {
          try {
            repo = (ref3 = JSON.parse(fs.readFileSync(path.join(packagePath, 'package.json')))) != null ? ref3.repository : void 0;
            repoUrl = (ref4 = repo != null ? repo.url : void 0) != null ? ref4 : repo;
          } catch (error) {}
        }
      }
      return repoUrl != null ? repoUrl.replace(/\.git$/, '').replace(/^git\+/, '') : void 0;
    };

    NotificationIssue.prototype.getPackageNameFromFilePath = function(filePath) {
      var packageName, ref, ref1, ref2, ref3;
      if (!filePath) {
        return;
      }
      packageName = (ref = /\/\.atom\/dev\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref1 = /\\\.atom\\dev\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref1[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref2 = /\/\.atom\/packages\/([^\/]+)\//.exec(filePath)) != null ? ref2[1] : void 0;
      if (packageName) {
        return packageName;
      }
      packageName = (ref3 = /\\\.atom\\packages\\([^\\]+)\\/.exec(filePath)) != null ? ref3[1] : void 0;
      if (packageName) {
        return packageName;
      }
    };

    NotificationIssue.prototype.getPackageName = function() {
      var file, getPackageName, i, j, options, packageName, packagePath, packagePaths, ref, stack;
      options = this.notification.getOptions();
      if (options.packageName != null) {
        return options.packageName;
      }
      if (!((options.stack != null) || (options.detail != null))) {
        return;
      }
      packagePaths = this.getPackagePathsByPackageName();
      for (packageName in packagePaths) {
        packagePath = packagePaths[packageName];
        if (packagePath.indexOf(path.join('.atom', 'dev', 'packages')) > -1 || packagePath.indexOf(path.join('.atom', 'packages')) > -1) {
          packagePaths[packageName] = fs.realpathSync(packagePath);
        }
      }
      getPackageName = (function(_this) {
        return function(filePath) {
          var isSubfolder, match, packName;
          filePath = /\((.+?):\d+|\((.+)\)|(.+)/.exec(filePath)[0];
          if (match = FileURLRegExp.exec(filePath)) {
            filePath = match[1];
          }
          filePath = path.normalize(filePath);
          if (path.isAbsolute(filePath)) {
            for (packName in packagePaths) {
              packagePath = packagePaths[packName];
              if (filePath === 'node.js') {
                continue;
              }
              isSubfolder = filePath.indexOf(path.normalize(packagePath + path.sep)) === 0;
              if (isSubfolder) {
                return packName;
              }
            }
          }
          return _this.getPackageNameFromFilePath(filePath);
        };
      })(this);
      if ((options.detail != null) && (packageName = getPackageName(options.detail))) {
        return packageName;
      }
      if (options.stack != null) {
        stack = StackTraceParser.parse(options.stack);
        for (i = j = 0, ref = stack.length; 0 <= ref ? j < ref : j > ref; i = 0 <= ref ? ++j : --j) {
          file = stack[i].file;
          if (!file) {
            return;
          }
          packageName = getPackageName(file);
          if (packageName != null) {
            return packageName;
          }
        }
      }
    };

    NotificationIssue.prototype.getPackagePathsByPackageName = function() {
      var j, len, pack, packagePathsByPackageName, ref;
      packagePathsByPackageName = {};
      ref = atom.packages.getLoadedPackages();
      for (j = 0, len = ref.length; j < len; j++) {
        pack = ref[j];
        packagePathsByPackageName[pack.name] = pack.path;
      }
      return packagePathsByPackageName;
    };

    return NotificationIssue;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
