(function() {
  var Highlights, _, cheerio, convertCodeBlocksToAtomEditors, fs, highlighter, packagePath, path, render, resolveImagePaths, resourcePath, roaster, sanitize, scopeForFenceName, tokenizeCodeBlocks;

  path = require('path');

  _ = require('underscore-plus');

  cheerio = require('cheerio');

  fs = require('fs-plus');

  Highlights = require('highlights');

  roaster = null;

  scopeForFenceName = require('./extension-helper').scopeForFenceName;

  highlighter = null;

  resourcePath = atom.getLoadSettings().resourcePath;

  packagePath = path.dirname(__dirname);

  exports.toDOMFragment = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, html) {
      var defaultCodeLanguage, domFragment, template;
      if (error != null) {
        return callback(error);
      }
      template = document.createElement('template');
      template.innerHTML = html;
      domFragment = template.content.cloneNode(true);
      if ((grammar != null ? grammar.scopeName : void 0) === 'source.litcoffee') {
        defaultCodeLanguage = 'coffee';
      }
      convertCodeBlocksToAtomEditors(domFragment, defaultCodeLanguage);
      return callback(null, domFragment);
    });
  };

  exports.toHTML = function(text, filePath, grammar, callback) {
    if (text == null) {
      text = '';
    }
    return render(text, filePath, function(error, html) {
      var defaultCodeLanguage;
      if (error != null) {
        return callback(error);
      }
      if ((grammar != null ? grammar.scopeName : void 0) === 'source.litcoffee') {
        defaultCodeLanguage = 'coffee';
      }
      html = tokenizeCodeBlocks(html, defaultCodeLanguage);
      return callback(null, html);
    });
  };

  render = function(text, filePath, callback) {
    var options;
    if (roaster == null) {
      roaster = require('roaster');
    }
    options = {
      sanitize: false,
      breaks: atom.config.get('markdown-preview.breakOnSingleNewline')
    };
    text = text.replace(/^\s*<!doctype(\s+.*)?>\s*/i, '');
    return roaster(text, options, function(error, html) {
      if (error != null) {
        return callback(error);
      }
      html = sanitize(html);
      html = resolveImagePaths(html, filePath);
      return callback(null, html.trim());
    });
  };

  sanitize = function(html) {
    var attribute, attributesToRemove, i, len, o;
    o = cheerio.load(html);
    o('script').remove();
    attributesToRemove = ['onabort', 'onblur', 'onchange', 'onclick', 'ondbclick', 'onerror', 'onfocus', 'onkeydown', 'onkeypress', 'onkeyup', 'onload', 'onmousedown', 'onmousemove', 'onmouseover', 'onmouseout', 'onmouseup', 'onreset', 'onresize', 'onscroll', 'onselect', 'onsubmit', 'onunload'];
    for (i = 0, len = attributesToRemove.length; i < len; i++) {
      attribute = attributesToRemove[i];
      o('*').removeAttr(attribute);
    }
    return o.html();
  };

  resolveImagePaths = function(html, filePath) {
    var i, img, imgElement, len, o, ref, rootDirectory, src;
    rootDirectory = atom.project.relativizePath(filePath)[0];
    o = cheerio.load(html);
    ref = o('img');
    for (i = 0, len = ref.length; i < len; i++) {
      imgElement = ref[i];
      img = o(imgElement);
      if (src = img.attr('src')) {
        if (src.match(/^(https?|atom):\/\//)) {
          continue;
        }
        if (src.startsWith(process.resourcesPath)) {
          continue;
        }
        if (src.startsWith(resourcePath)) {
          continue;
        }
        if (src.startsWith(packagePath)) {
          continue;
        }
        if (src[0] === '/') {
          if (!fs.isFileSync(src)) {
            if (rootDirectory) {
              img.attr('src', path.join(rootDirectory, src.substring(1)));
            }
          }
        } else {
          img.attr('src', path.resolve(path.dirname(filePath), src));
        }
      }
    }
    return o.html();
  };

  convertCodeBlocksToAtomEditors = function(domFragment, defaultLanguage) {
    var codeBlock, codeElement, cursorLineDecoration, editor, editorElement, fenceName, fontFamily, grammar, i, j, k, len, len1, len2, preElement, ref, ref1, ref2, ref3, ref4, ref5;
    if (defaultLanguage == null) {
      defaultLanguage = 'text';
    }
    if (fontFamily = atom.config.get('editor.fontFamily')) {
      ref = domFragment.querySelectorAll('code');
      for (i = 0, len = ref.length; i < len; i++) {
        codeElement = ref[i];
        codeElement.style.fontFamily = fontFamily;
      }
    }
    ref1 = domFragment.querySelectorAll('pre');
    for (j = 0, len1 = ref1.length; j < len1; j++) {
      preElement = ref1[j];
      codeBlock = (ref2 = preElement.firstElementChild) != null ? ref2 : preElement;
      fenceName = (ref3 = (ref4 = codeBlock.getAttribute('class')) != null ? ref4.replace(/^lang-/, '') : void 0) != null ? ref3 : defaultLanguage;
      editorElement = document.createElement('atom-text-editor');
      editorElement.setAttributeNode(document.createAttribute('gutter-hidden'));
      editorElement.removeAttribute('tabindex');
      preElement.parentNode.insertBefore(editorElement, preElement);
      preElement.remove();
      editor = editorElement.getModel();
      editor.setText(codeBlock.textContent);
      if (grammar = atom.grammars.grammarForScopeName(scopeForFenceName(fenceName))) {
        editor.setGrammar(grammar);
      }
      if (editor.cursorLineDecorations != null) {
        ref5 = editor.cursorLineDecorations;
        for (k = 0, len2 = ref5.length; k < len2; k++) {
          cursorLineDecoration = ref5[k];
          cursorLineDecoration.destroy();
        }
      } else {
        editor.getDecorations({
          "class": 'cursor-line',
          type: 'line'
        })[0].destroy();
      }
    }
    return domFragment;
  };

  tokenizeCodeBlocks = function(html, defaultLanguage) {
    var codeBlock, fenceName, fontFamily, highlightedBlock, highlightedHtml, i, len, o, preElement, ref, ref1, ref2;
    if (defaultLanguage == null) {
      defaultLanguage = 'text';
    }
    o = cheerio.load(html);
    if (fontFamily = atom.config.get('editor.fontFamily')) {
      o('code').css('font-family', fontFamily);
    }
    ref = o("pre");
    for (i = 0, len = ref.length; i < len; i++) {
      preElement = ref[i];
      codeBlock = o(preElement).children().first();
      fenceName = (ref1 = (ref2 = codeBlock.attr('class')) != null ? ref2.replace(/^lang-/, '') : void 0) != null ? ref1 : defaultLanguage;
      if (highlighter == null) {
        highlighter = new Highlights({
          registry: atom.grammars,
          scopePrefix: 'syntax--'
        });
      }
      highlightedHtml = highlighter.highlightSync({
        fileContents: codeBlock.text(),
        scopeName: scopeForFenceName(fenceName)
      });
      highlightedBlock = o(highlightedHtml);
      highlightedBlock.removeClass('editor').addClass("lang-" + fenceName);
      o(preElement).replaceWith(highlightedBlock);
    }
    return o.html();
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
