'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _listSelection = require('./list-selection');

var _listSelection2 = _interopRequireDefault(_listSelection);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const COPY = {};

let FilePatchSelection = class FilePatchSelection {
  constructor(hunks) {
    if (hunks._copy !== COPY) {
      // Initialize a new selection
      this.mode = 'hunk';

      this.hunksByLine = new Map();
      const lines = [];
      for (const hunk of hunks) {
        for (const line of hunk.lines) {
          lines.push(line);
          this.hunksByLine.set(line, hunk);
        }
      }

      this.hunksSelection = new _listSelection2.default({ items: hunks });
      this.linesSelection = new _listSelection2.default({ items: lines, isItemSelectable: line => line.isChanged() });
      this.resolveNextUpdatePromise = () => {};
    } else {
      // Copy from options. *Only* reachable from the copy() method because no other module has visibility to
      // the COPY object without shenanigans.
      const options = hunks;

      this.mode = options.mode;
      this.hunksSelection = options.hunksSelection;
      this.linesSelection = options.linesSelection;
      this.resolveNextUpdatePromise = options.resolveNextUpdatePromise;
      this.hunksByLine = options.hunksByLine;
    }
  }

  copy() {
    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    const mode = options.mode || this.mode;
    const hunksSelection = options.hunksSelection || this.hunksSelection.copy();
    const linesSelection = options.linesSelection || this.linesSelection.copy();

    let hunksByLine = null;
    if (options.hunks) {
      // Update hunks
      const oldHunks = this.hunksSelection.getItems();
      const newHunks = options.hunks;

      let wasChanged = false;
      if (newHunks.length !== oldHunks.length) {
        wasChanged = true;
      } else {
        for (let i = 0; i < oldHunks.length; i++) {
          if (oldHunks[i] !== newHunks[i]) {
            wasChanged = true;
            break;
          }
        }
      }

      // Update hunks, preserving selection index
      hunksSelection.setItems(newHunks);

      const oldLines = this.linesSelection.getItems();
      const newLines = [];

      hunksByLine = new Map();
      for (const hunk of newHunks) {
        for (const line of hunk.lines) {
          newLines.push(line);
          hunksByLine.set(line, hunk);
        }
      }

      // Update lines, preserving selection index in *changed* lines
      let newSelectedLine;
      if (oldLines.length > 0 && newLines.length > 0) {
        const oldSelectionStartIndex = this.linesSelection.getMostRecentSelectionStartIndex();
        let changedLineCount = 0;
        for (let i = 0; i < oldSelectionStartIndex; i++) {
          if (oldLines[i].isChanged()) {
            changedLineCount++;
          }
        }

        for (let i = 0; i < newLines.length; i++) {
          const line = newLines[i];
          if (line.isChanged()) {
            newSelectedLine = line;
            if (changedLineCount === 0) {
              break;
            }
            changedLineCount--;
          }
        }
      }

      linesSelection.setItems(newLines);
      if (newSelectedLine) {
        linesSelection.selectItem(newSelectedLine);
      }
      if (wasChanged) {
        this.resolveNextUpdatePromise();
      }
    } else {
      // Hunks are unchanged. Don't recompute hunksByLine.
      hunksByLine = this.hunksByLine;
    }

    return new FilePatchSelection({
      _copy: COPY,
      mode,
      hunksSelection,
      linesSelection,
      hunksByLine,
      resolveNextUpdatePromise: options.resolveNextUpdatePromise || this.resolveNextUpdatePromise
    });
  }

  toggleMode() {
    if (this.mode === 'hunk') {
      const firstLineOfSelectedHunk = this.getHeadHunk().lines[0];
      const selection = this.selectLine(firstLineOfSelectedHunk);
      if (!firstLineOfSelectedHunk.isChanged()) {
        return selection.selectNextLine();
      } else {
        return selection;
      }
    } else {
      const selectedLine = this.getHeadLine();
      const hunkContainingSelectedLine = this.hunksByLine.get(selectedLine);
      return this.selectHunk(hunkContainingSelectedLine);
    }
  }

  getMode() {
    return this.mode;
  }

  selectNext() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (this.mode === 'hunk') {
      return this.selectNextHunk(preserveTail);
    } else {
      return this.selectNextLine(preserveTail);
    }
  }

  selectPrevious() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    if (this.mode === 'hunk') {
      return this.selectPreviousHunk(preserveTail);
    } else {
      return this.selectPreviousLine(preserveTail);
    }
  }

  selectAll() {
    if (this.mode === 'hunk') {
      return this.selectAllHunks();
    } else {
      return this.selectAllLines();
    }
  }

  selectFirst(preserveTail) {
    if (this.mode === 'hunk') {
      return this.selectFirstHunk(preserveTail);
    } else {
      return this.selectFirstLine(preserveTail);
    }
  }

  selectLast(preserveTail) {
    if (this.mode === 'hunk') {
      return this.selectLastHunk(preserveTail);
    } else {
      return this.selectLastLine(preserveTail);
    }
  }

  selectHunk(hunk) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectItem(hunk, preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  addOrSubtractHunkSelection(hunk) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.addOrSubtractSelection(hunk);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectAllHunks() {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectAllItems();

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectFirstHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectFirstItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectLastHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectLastItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  jumpToNextHunk() {
    const next = this.selectNextHunk();
    return next.getMode() !== this.mode ? next.toggleMode() : next;
  }

  jumpToPreviousHunk() {
    const next = this.selectPreviousHunk();
    return next.getMode() !== this.mode ? next.toggleMode() : next;
  }

  selectNextHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectNextItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  selectPreviousHunk(preserveTail) {
    const hunksSelection = this.hunksSelection.copy();
    hunksSelection.selectPreviousItem(preserveTail);

    return this.copy({ mode: 'hunk', hunksSelection });
  }

  getSelectedHunks() {
    if (this.mode === 'line') {
      const selectedHunks = new Set();
      const selectedLines = this.getSelectedLines();
      selectedLines.forEach(line => selectedHunks.add(this.hunksByLine.get(line)));
      return selectedHunks;
    } else {
      return this.hunksSelection.getSelectedItems();
    }
  }

  getHeadHunk() {
    return this.mode === 'hunk' ? this.hunksSelection.getHeadItem() : null;
  }

  selectLine(line) {
    let preserveTail = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectItem(line, preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  addOrSubtractLineSelection(line) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.addOrSubtractSelection(line);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectAllLines(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectAllItems(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectFirstLine(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectFirstItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectLastLine(preserveTail) {
    const linesSelection = this.linesSelection.copy();
    linesSelection.selectLastItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectNextLine() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectNextItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  selectPreviousLine() {
    let preserveTail = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : false;

    const linesSelection = this.linesSelection.copy();
    linesSelection.selectPreviousItem(preserveTail);
    return this.copy({ mode: 'line', linesSelection });
  }

  getSelectedLines() {
    if (this.mode === 'hunk') {
      const selectedLines = new Set();
      this.getSelectedHunks().forEach(hunk => {
        for (const line of hunk.lines) {
          if (line.isChanged()) {
            selectedLines.add(line);
          }
        }
      });
      return selectedLines;
    } else {
      return this.linesSelection.getSelectedItems();
    }
  }

  getHeadLine() {
    return this.mode === 'line' ? this.linesSelection.getHeadItem() : null;
  }

  updateHunks(newHunks) {
    return this.copy({ hunks: newHunks });
  }

  coalesce() {
    const hunksSelection = this.hunksSelection.copy();
    const linesSelection = this.linesSelection.copy();

    hunksSelection.coalesce();
    linesSelection.coalesce();

    return this.copy({ hunksSelection, linesSelection });
  }

  getNextUpdatePromise() {
    return new Promise((resolve, reject) => {
      this.resolveNextUpdatePromise = resolve;
    });
  }

  getLineSelectionTailIndex() {
    return this.linesSelection.getTailIndex();
  }

  goToDiffLine(lineNumber) {
    const lines = this.linesSelection.getItems();

    let closestLine;
    let closestLineDistance = Infinity;

    for (let i = 0; i < lines.length; i++) {
      const line = lines[i];
      if (!this.linesSelection.isItemSelectable(line)) {
        continue;
      }
      if (line.newLineNumber === lineNumber) {
        return this.selectLine(line);
      } else {
        const newDistance = Math.abs(line.newLineNumber - lineNumber);
        if (newDistance < closestLineDistance) {
          closestLineDistance = newDistance;
          closestLine = line;
        } else {
          return this.selectLine(closestLine);
        }
      }
    }

    return this.selectLine(closestLine);
  }
};
exports.default = FilePatchSelection;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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