'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = undefined;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _fs = require('fs');

var _fs2 = _interopRequireDefault(_fs);

var _helpers = require('../helpers');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

/**
 * Locate the nearest git working directory above a given starting point, caching results.
 */
let WorkdirCache = class WorkdirCache {
  constructor() {
    let maxSize = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 1000;

    this.maxSize = maxSize;
    this.known = new Map();
  }

  find(startDir) {
    var _this = this;

    return _asyncToGenerator(function* () {
      try {
        const resolvedDir = yield _this.resolvePath(startDir);
        const cached = _this.known.get(resolvedDir);
        if (cached !== undefined) {
          return cached;
        }

        const workDir = yield _this.walkToRoot(resolvedDir);

        if (_this.known.size >= _this.maxSize) {
          _this.known.clear();
        }
        _this.known.set(resolvedDir, workDir);
        return workDir;
      } catch (e) {
        if (e.code === 'ENOENT') {
          return null;
        }

        throw e;
      }
    })();
  }

  invalidate(baseDir) {
    var _this2 = this;

    return _asyncToGenerator(function* () {
      const resolvedBase = yield _this2.resolvePath(baseDir);
      for (const cachedPath of _this2.known.keys()) {
        if (cachedPath.startsWith(resolvedBase)) {
          _this2.known.delete(cachedPath);
        }
      }
    })();
  }

  resolvePath(unresolvedPath) {
    return new Promise((resolve, reject) => {
      _fs2.default.realpath(unresolvedPath, (err, resolved) => err ? reject(err) : resolve(resolved));
    });
  }

  walkToRoot(initialDir) {
    return new Promise((resolve, reject) => {
      let currentDir = initialDir;

      const check = () => {
        const dotGit = _path2.default.join(currentDir, '.git');
        _fs2.default.stat(dotGit, (() => {
          var _ref = _asyncToGenerator(function* (statError, stat) {
            if (statError) {
              if (statError.code === 'ENOENT' || statError.code === 'ENOTDIR') {
                // File not found. This is not the directory we're looking for. Continue walking.
                return walk();
              }

              return reject(statError);
            }

            if (!stat.isDirectory()) {
              const contents = yield (0, _helpers.readFile)(dotGit, 'utf8');
              if (contents.startsWith('gitdir: ')) {
                return resolve(currentDir);
              } else {
                return walk();
              }
            }

            // .git directory found! Mission accomplished.
            return resolve(currentDir);
          });

          return function (_x2, _x3) {
            return _ref.apply(this, arguments);
          };
        })());
      };

      const walk = () => {
        const parentDir = _path2.default.resolve(currentDir, '..');
        if (parentDir === currentDir) {
          // Root directory. Traversal done, no working directory found.
          resolve(null);
          return;
        }

        currentDir = parentDir;
        check();
      };

      check();
    });
  }
};
exports.default = WorkdirCache;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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