'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.CommitError = exports.AbortMergeError = undefined;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _eventKit = require('event-kit');

var _compositeGitStrategy = require('../composite-git-strategy');

var _compositeGitStrategy2 = _interopRequireDefault(_compositeGitStrategy);

var _helpers = require('../helpers');

var _remote = require('./remote');

var _remote2 = _interopRequireDefault(_remote);

var _repositoryStates = require('./repository-states');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

const MERGE_MARKER_REGEX = /^(>|<){7} \S+$/m;

// Internal option keys used to designate the desired initial state of a Repository.
const initialStateSym = Symbol('initialState');

let AbortMergeError = exports.AbortMergeError = class AbortMergeError extends Error {
  constructor(code, filePath) {
    super();
    this.message = `${code}: ${filePath}.`;
    this.code = code;
    this.path = filePath;
    this.stack = new Error().stack;
  }
};
let CommitError = exports.CommitError = class CommitError extends Error {
  constructor(code) {
    super();
    this.message = `Commit error: ${code}.`;
    this.code = code;
    this.stack = new Error().stack;
  }
};
let Repository = class Repository {
  constructor(workingDirectoryPath) {
    let gitStrategy = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : null;
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    this.workingDirectoryPath = workingDirectoryPath;
    this.git = gitStrategy || _compositeGitStrategy2.default.create(workingDirectoryPath);

    this.emitter = new _eventKit.Emitter();

    this.loadPromise = new Promise(resolve => {
      const sub = this.onDidChangeState(() => {
        if (!this.isLoading()) {
          resolve();
          sub.dispose();
        } else if (this.isDestroyed()) {
          sub.dispose();
        }
      });
    });

    this.transitionTo(options[initialStateSym] || _repositoryStates.Loading);
  }

  static absent() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.Absent });
  }

  static loadingGuess() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.LoadingGuess });
  }

  static absentGuess() {
    return new Repository(null, null, { [initialStateSym]: _repositoryStates.AbsentGuess });
  }

  // State management //////////////////////////////////////////////////////////////////////////////////////////////////

  transition(currentState, StateConstructor) {
    if (currentState !== this.state) {
      // Attempted transition from a non-active state, most likely from an asynchronous start() method.
      return Promise.resolve();
    }

    for (var _len = arguments.length, payload = Array(_len > 2 ? _len - 2 : 0), _key = 2; _key < _len; _key++) {
      payload[_key - 2] = arguments[_key];
    }

    const nextState = new StateConstructor(this, ...payload);
    this.state = nextState;

    this.emitter.emit('did-change-state', { from: currentState, to: this.state });
    if (!this.isDestroyed()) {
      this.emitter.emit('did-update');
    }

    return this.state.start();
  }

  transitionTo(StateConstructor) {
    for (var _len2 = arguments.length, payload = Array(_len2 > 1 ? _len2 - 1 : 0), _key2 = 1; _key2 < _len2; _key2++) {
      payload[_key2 - 1] = arguments[_key2];
    }

    return this.transition(this.state, StateConstructor, ...payload);
  }

  getLoadPromise() {
    return this.isAbsent() ? Promise.reject(new Error('An absent repository will never load')) : this.loadPromise;
  }

  /*
   * Use `callback` to request user input from all git strategies.
   */
  setPromptCallback(callback) {
    this.git.getImplementers().forEach(strategy => strategy.setPromptCallback(callback));
  }

  // Event subscription ////////////////////////////////////////////////////////////////////////////////////////////////

  onDidDestroy(callback) {
    return this.emitter.on('did-destroy', callback);
  }

  onDidChangeState(callback) {
    return this.emitter.on('did-change-state', callback);
  }

  onDidUpdate(callback) {
    return this.emitter.on('did-update', callback);
  }

  // State-independent actions /////////////////////////////////////////////////////////////////////////////////////////
  // Actions that use direct filesystem access or otherwise don't need `this.git` to be available.

  pathHasMergeMarkers(relativePath) {
    var _this = this;

    return _asyncToGenerator(function* () {
      try {
        const contents = yield (0, _helpers.readFile)(_path2.default.join(_this.getWorkingDirectoryPath(), relativePath), 'utf8');
        return MERGE_MARKER_REGEX.test(contents);
      } catch (e) {
        // EISDIR implies this is a submodule
        if (e.code === 'ENOENT' || e.code === 'EISDIR') {
          return false;
        } else {
          throw e;
        }
      }
    })();
  }

  getMergeMessage() {
    var _this2 = this;

    return _asyncToGenerator(function* () {
      try {
        const contents = yield (0, _helpers.readFile)(_path2.default.join(_this2.getGitDirectoryPath(), 'MERGE_MSG'), 'utf8');
        return contents;
      } catch (e) {
        return null;
      }
    })();
  }

  // State-independent accessors ///////////////////////////////////////////////////////////////////////////////////////

  getWorkingDirectoryPath() {
    return this.workingDirectoryPath;
  }

  setGitDirectoryPath(gitDirectoryPath) {
    this._gitDirectoryPath = gitDirectoryPath;
  }

  getGitDirectoryPath() {
    if (this._gitDirectoryPath) {
      return this._gitDirectoryPath;
    } else if (this.getWorkingDirectoryPath()) {
      return _path2.default.join(this.getWorkingDirectoryPath(), '.git');
    } else {
      // Absent/Loading/etc.
      return null;
    }
  }

  isInState(stateName) {
    return this.state.constructor.name === stateName;
  }

  toString() {
    return `Repository(state=${this.state.constructor.name}, workdir="${this.getWorkingDirectoryPath()}")`;
  }

  // Compound Getters //////////////////////////////////////////////////////////////////////////////////////////////////
  // Accessor methods for data derived from other, state-provided getters.

  getUnstagedChanges() {
    var _this3 = this;

    return _asyncToGenerator(function* () {
      var _ref = yield _this3.getStatusBundle();

      const unstagedFiles = _ref.unstagedFiles;

      return Object.keys(unstagedFiles).map(function (filePath) {
        return { filePath, status: unstagedFiles[filePath] };
      });
    })();
  }

  getStagedChanges() {
    var _this4 = this;

    return _asyncToGenerator(function* () {
      var _ref2 = yield _this4.getStatusBundle();

      const stagedFiles = _ref2.stagedFiles;

      return Object.keys(stagedFiles).map(function (filePath) {
        return { filePath, status: stagedFiles[filePath] };
      });
    })();
  }

  getMergeConflicts() {
    var _this5 = this;

    return _asyncToGenerator(function* () {
      var _ref3 = yield _this5.getStatusBundle();

      const mergeConflictFiles = _ref3.mergeConflictFiles;

      return Object.keys(mergeConflictFiles).map(function (filePath) {
        return { filePath, status: mergeConflictFiles[filePath] };
      });
    })();
  }

  isPartiallyStaged(fileName) {
    var _this6 = this;

    return _asyncToGenerator(function* () {
      var _ref4 = yield _this6.getStatusBundle();

      const unstagedFiles = _ref4.unstagedFiles,
            stagedFiles = _ref4.stagedFiles;

      const u = unstagedFiles[fileName];
      const s = stagedFiles[fileName];
      return u === 'modified' && s === 'modified' || u === 'modified' && s === 'added' || u === 'added' && s === 'deleted' || u === 'deleted' && s === 'modified';
    })();
  }

  getRemoteForBranch(branchName) {
    var _this7 = this;

    return _asyncToGenerator(function* () {
      const name = yield _this7.getConfig(`branch.${branchName}.remote`);
      if (name === null) {
        return _remote.nullRemote;
      } else {
        return new _remote2.default(name);
      }
    })();
  }

  saveDiscardHistory() {
    var _this8 = this;

    return _asyncToGenerator(function* () {
      if (_this8.isDestroyed()) {
        return;
      }

      const historySha = yield _this8.createDiscardHistoryBlob();
      if (_this8.isDestroyed()) {
        return;
      }
      yield _this8.setConfig('atomGithub.historySha', historySha);
    })();
  }
};

// The methods named here will be delegated to the current State.
//
// This list should match the methods decorated with @shouldDelegate in `lib/models/repository-states/state.js`. A test
// case in `test/models/repository.test.js` ensures that these sets match.
//
// Duplicated here rather than just using `expectedDelegates` directly so that this file is grep-friendly for answering
// the question of "what all can a Repository do exactly".

exports.default = Repository;
const delegates = ['isLoadingGuess', 'isAbsentGuess', 'isAbsent', 'isLoading', 'isEmpty', 'isPresent', 'isDestroyed', 'isUndetermined', 'showGitTabInit', 'showGitTabInitInProgress', 'showGitTabLoading', 'showStatusBarTiles', 'hasDirectory', 'init', 'clone', 'destroy', 'refresh', 'observeFilesystemChange', 'stageFiles', 'unstageFiles', 'stageFilesFromParentCommit', 'applyPatchToIndex', 'applyPatchToWorkdir', 'commit', 'merge', 'abortMerge', 'checkoutSide', 'mergeFile', 'writeMergeConflictToIndex', 'checkout', 'checkoutPathsAtRevision', 'fetch', 'pull', 'push', 'setConfig', 'createBlob', 'expandBlobToFile', 'createDiscardHistoryBlob', 'updateDiscardHistory', 'storeBeforeAndAfterBlobs', 'restoreLastDiscardInTempFiles', 'popDiscardHistory', 'clearDiscardHistory', 'discardWorkDirChangesForPaths', 'getStatusBundle', 'getStatusesForChangedFiles', 'getStagedChangesSinceParentCommit', 'getFilePatchForPath', 'readFileFromIndex', 'getLastCommit', 'getBranches', 'getCurrentBranch', 'isMerging', 'isRebasing', 'getRemotes', 'getAheadCount', 'getBehindCount', 'getConfig', 'unsetConfig', 'getBlobContents', 'hasDiscardHistory', 'getDiscardHistory', 'getLastHistorySnapshots'];

for (let i = 0; i < delegates.length; i++) {
  const delegate = delegates[i];

  Repository.prototype[delegate] = function () {
    return this.state[delegate](...arguments);
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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