'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = exports.GitError = undefined;

var _slicedToArray = function () { function sliceIterator(arr, i) { var _arr = []; var _n = true; var _d = false; var _e = undefined; try { for (var _i = arr[Symbol.iterator](), _s; !(_n = (_s = _i.next()).done); _n = true) { _arr.push(_s.value); if (i && _arr.length === i) break; } } catch (err) { _d = true; _e = err; } finally { try { if (!_n && _i["return"]) _i["return"](); } finally { if (_d) throw _e; } } return _arr; } return function (arr, i) { if (Array.isArray(arr)) { return arr; } else if (Symbol.iterator in Object(arr)) { return sliceIterator(arr, i); } else { throw new TypeError("Invalid attempt to destructure non-iterable instance"); } }; }();

var _class, _temp;

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _os = require('os');

var _os2 = _interopRequireDefault(_os);

var _child_process = require('child_process');

var _child_process2 = _interopRequireDefault(_child_process);

var _electron = require('electron');

var _eventKit = require('event-kit');

var _dugite = require('dugite');

var _whatTheDiff = require('what-the-diff');

var _whatTheStatus = require('what-the-status');

var _gitPromptServer = require('./git-prompt-server');

var _gitPromptServer2 = _interopRequireDefault(_gitPromptServer);

var _asyncQueue = require('./async-queue');

var _asyncQueue2 = _interopRequireDefault(_asyncQueue);

var _helpers = require('./helpers');

var _gitTimingsView = require('./views/git-timings-view');

var _gitTimingsView2 = _interopRequireDefault(_gitTimingsView);

var _workerManager = require('./worker-manager');

var _workerManager2 = _interopRequireDefault(_workerManager);

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _asyncToGenerator(fn) { return function () { var gen = fn.apply(this, arguments); return new Promise(function (resolve, reject) { function step(key, arg) { try { var info = gen[key](arg); var value = info.value; } catch (error) { reject(error); return; } if (info.done) { resolve(value); } else { return Promise.resolve(value).then(function (value) { step("next", value); }, function (err) { step("throw", err); }); } } return step("next"); }); }; }

const LINE_ENDING_REGEX = /\r?\n/;

const GPG_HELPER_PATH = _path2.default.resolve((0, _helpers.getPackageRoot)(), 'bin', 'gpg-no-tty.sh');
const ENV_VARS_TO_COPY = ['GIT_AUTHOR_NAME', 'GIT_AUTHOR_EMAIL', 'GIT_AUTHOR_DATE', 'GIT_COMMITTER_NAME', 'GIT_COMMITTER_EMAIL', 'GIT_COMMITTER_DATE', 'EMAIL'];

let headless = null;
let execPathPromise = null;

let GitError = exports.GitError = class GitError extends Error {
  constructor(message) {
    super(message);
    this.message = message;
    this.stack = new Error().stack;
  }
};
let GitShellOutStrategy = (_temp = _class = class GitShellOutStrategy {

  constructor(workingDir) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    this.workingDir = workingDir;
    if (options.queue) {
      this.commandQueue = options.queue;
    } else {
      const parallelism = options.parallelism || Math.max(3, _os2.default.cpus().length);
      this.commandQueue = new _asyncQueue2.default({ parallelism });
    }

    this.prompt = options.prompt || (query => Promise.reject());
    this.workerManager = options.workerManager;

    if (headless === null) {
      headless = !_electron.remote.getCurrentWindow().isVisible();
    }
  }

  /*
   * Provide an asynchronous callback to be used to request input from the user for git operations.
   *
   * `prompt` must be a callable that accepts a query object `{prompt, includeUsername}` and returns a Promise
   * that either resolves with a result object `{[username], password}` or rejects on cancellation.
   */
  setPromptCallback(prompt) {
    this.prompt = prompt;
  }

  // Execute a command and read the output using the embedded Git environment
  exec(args) {
    var _this = this;

    var _ref = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : GitShellOutStrategy.defaultExecArgs;

    let stdin = _ref.stdin,
        useGitPromptServer = _ref.useGitPromptServer,
        writeOperation = _ref.writeOperation;
    return _asyncToGenerator(function* () {
      /* eslint-disable no-console */
      const subscriptions = new _eventKit.CompositeDisposable();
      const diagnosticsEnabled = process.env.ATOM_GITHUB_GIT_DIAGNOSTICS || atom.config.get('github.gitDiagnostics');

      const formattedArgs = `git ${args.join(' ')} in ${_this.workingDir}`;
      const timingMarker = _gitTimingsView2.default.generateMarker(`git ${args.join(' ')}`);
      timingMarker.mark('queued');

      if (execPathPromise === null) {
        // Attempt to collect the --exec-path from a native git installation.
        execPathPromise = new Promise(function (resolve, reject) {
          _child_process2.default.exec('git --exec-path', function (error, stdout, stderr) {
            if (error) {
              // Oh well
              resolve(null);
              return;
            }

            resolve(stdout.trim());
          });
        });
      }
      const execPath = yield execPathPromise;

      return _this.commandQueue.push(_asyncToGenerator(function* () {
        timingMarker.mark('prepare');
        let gitPromptServer;

        const pathParts = [];
        if (process.env.PATH) {
          pathParts.push(process.env.PATH);
        }
        if (execPath) {
          pathParts.push(execPath);
        }

        const env = {
          GIT_TERMINAL_PROMPT: '0',
          PATH: pathParts.join(_path2.default.delimiter)
        };

        ENV_VARS_TO_COPY.forEach(function (envVar) {
          if (process.env[envVar] !== undefined) {
            env[envVar] = process.env[envVar];
          }
        });

        if (useGitPromptServer) {
          gitPromptServer = new _gitPromptServer2.default();

          var _ref3 = yield gitPromptServer.start(_this.prompt);

          const socket = _ref3.socket,
                electron = _ref3.electron,
                credentialHelper = _ref3.credentialHelper,
                askPass = _ref3.askPass,
                sshWrapper = _ref3.sshWrapper;


          env.ATOM_GITHUB_ASKPASS_PATH = (0, _helpers.normalizeGitHelperPath)(askPass.script);
          env.ATOM_GITHUB_CREDENTIAL_PATH = (0, _helpers.normalizeGitHelperPath)(credentialHelper.script);
          env.ATOM_GITHUB_ELECTRON_PATH = (0, _helpers.normalizeGitHelperPath)(electron);
          env.ATOM_GITHUB_SOCK_PATH = (0, _helpers.normalizeGitHelperPath)(socket);

          env.ATOM_GITHUB_WORKDIR_PATH = _this.workingDir;
          env.ATOM_GITHUB_DUGITE_PATH = (0, _helpers.getDugitePath)();

          // "ssh" won't respect SSH_ASKPASS unless:
          // (a) it's running without a tty
          // (b) DISPLAY is set to something nonempty
          // But, on a Mac, DISPLAY is unset. Ensure that it is so our SSH_ASKPASS is respected.
          if (!process.env.DISPLAY || process.env.DISPLAY.length === 0) {
            env.DISPLAY = 'atom-github-placeholder';
          }

          env.ATOM_GITHUB_ORIGINAL_PATH = process.env.PATH || '';
          env.ATOM_GITHUB_ORIGINAL_GIT_ASKPASS = process.env.GIT_ASKPASS || '';
          env.ATOM_GITHUB_ORIGINAL_SSH_ASKPASS = process.env.SSH_ASKPASS || '';
          env.ATOM_GITHUB_ORIGINAL_GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND || '';
          env.ATOM_GITHUB_SPEC_MODE = atom.inSpecMode() ? 'true' : 'false';

          env.SSH_ASKPASS = (0, _helpers.normalizeGitHelperPath)(askPass.launcher);
          env.GIT_ASKPASS = (0, _helpers.normalizeGitHelperPath)(askPass.launcher);

          if (process.platform === 'linux') {
            env.GIT_SSH_COMMAND = sshWrapper.script;
          } else {
            env.GIT_SSH_COMMAND = process.env.GIT_SSH_COMMAND;
          }

          args.unshift('-c', `credential.helper=${(0, _helpers.normalizeGitHelperPath)(credentialHelper.launcher)}`);
        }

        if (diagnosticsEnabled) {
          env.GIT_TRACE = 'true';
          env.GIT_TRACE_CURL = 'true';
        }

        const options = { env };

        if (stdin) {
          options.stdin = stdin;
          options.stdinEncoding = 'utf8';
        }

        if (process.env.PRINT_GIT_TIMES) {
          console.time(`git:${formattedArgs}`);
        }
        return new Promise((() => {
          var _ref4 = _asyncToGenerator(function* (resolve, reject) {
            var _executeGitCommand = _this.executeGitCommand(args, options, timingMarker);

            const promise = _executeGitCommand.promise,
                  cancel = _executeGitCommand.cancel;

            let expectCancel = false;
            if (gitPromptServer) {
              subscriptions.add(gitPromptServer.onDidCancel((() => {
                var _ref5 = _asyncToGenerator(function* (_ref6) {
                  let handlerPid = _ref6.handlerPid;

                  expectCancel = true;
                  yield cancel();

                  // On Windows, the SSH_ASKPASS handler is executed as a non-child process, so the bin\git-askpass-atom.sh
                  // process does not terminate when the git process is killed.
                  // Kill the handler process *after* the git process has been killed to ensure that git doesn't have a
                  // chance to fall back to GIT_ASKPASS from the credential handler.
                  require('tree-kill')(handlerPid);
                });

                return function (_x5) {
                  return _ref5.apply(this, arguments);
                };
              })()));
            }

            var _ref7 = yield promise;

            const stdout = _ref7.stdout,
                  stderr = _ref7.stderr,
                  exitCode = _ref7.exitCode,
                  timing = _ref7.timing;


            if (timing) {
              const execTime = timing.execTime,
                    spawnTime = timing.spawnTime,
                    ipcTime = timing.ipcTime;

              const now = performance.now();
              timingMarker.mark('nexttick', now - execTime - spawnTime - ipcTime);
              timingMarker.mark('execute', now - execTime - ipcTime);
              timingMarker.mark('ipc', now - ipcTime);
            }
            timingMarker.finalize();
            if (process.env.PRINT_GIT_TIMES) {
              console.timeEnd(`git:${formattedArgs}`);
            }
            if (gitPromptServer) {
              gitPromptServer.terminate();
            }
            subscriptions.dispose();

            if (diagnosticsEnabled) {
              if (headless) {
                let summary = `git:${formattedArgs}\n`;
                summary += `exit status: ${exitCode}\n`;
                summary += 'stdout:';
                if (stdout.length === 0) {
                  summary += ' <empty>\n';
                } else {
                  summary += `\n${stdout}\n`;
                }
                summary += 'stderr:';
                if (stderr.length === 0) {
                  summary += ' <empty>\n';
                } else {
                  summary += `\n${stderr}\n`;
                }

                console.log(summary);
              } else {
                const headerStyle = 'font-weight: bold; color: blue;';

                console.groupCollapsed(`git:${formattedArgs}`);
                console.log('%cexit status%c %d', headerStyle, 'font-weight: normal; color: black;', exitCode);
                console.log('%cstdout', headerStyle);
                console.log(stdout);
                console.log('%cstderr', headerStyle);
                console.log(stderr);
                console.groupEnd();
              }
            }

            if (exitCode !== 0 && !expectCancel) {
              const err = new GitError(`${formattedArgs} exited with code ${exitCode}\nstdout: ${stdout}\nstderr: ${stderr}`);
              err.code = exitCode;
              err.stdErr = stderr;
              err.stdOut = stdout;
              err.command = formattedArgs;
              reject(err);
            }
            resolve(stdout);
          });

          return function (_x3, _x4) {
            return _ref4.apply(this, arguments);
          };
        })());
      }), { parallel: !writeOperation });
      /* eslint-enable no-console */
    })();
  }

  executeGitCommand(args, options) {
    let marker = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : null;

    if (process.env.ATOM_GITHUB_INLINE_GIT_EXEC || !_workerManager2.default.getInstance().isReady()) {
      marker && marker.mark('nexttick');

      let childPid;
      options.processCallback = child => {
        childPid = child.pid;

        child.on('error', err => {
          /* eslint-disable no-console */
          console.error(`Error spawning: git ${args.join(' ')} in ${this.workingDir}`);
          console.error(err);
          /* eslint-enable no-console */
        });

        child.stdin.on('error', err => {
          /* eslint-disable no-console */
          console.error(`Error writing to stdin: git ${args.join(' ')} in ${this.workingDir}\n${options.stdin}`);
          console.error(err);
          /* eslint-enable no-console */
        });
      };

      const promise = _dugite.GitProcess.exec(args, this.workingDir, options);
      marker && marker.mark('execute');
      return {
        promise,
        cancel: () => childPid && require('tree-kill')(childPid)
      };
    } else {
      const workerManager = this.workerManager || _workerManager2.default.getInstance();
      return workerManager.request({
        args,
        workingDir: this.workingDir,
        options
      });
    }
  }

  /**
   * Execute a git command that may create a commit. If the command fails because the GPG binary was invoked and unable
   * to acquire a passphrase (because the pinentry program attempted to use a tty), retry with a `GitPromptServer`.
   */
  gpgExec(args) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const gpgArgs = ['-c', `gpg.program=${GPG_HELPER_PATH}`].concat(args);
    return this.exec(gpgArgs, options).catch(err => {
      if (err.code === 128 && /gpg failed/.test(err.stdErr) && !options.useGitPromptServer) {
        // Retry with a GitPromptServer
        options.useGitPromptServer = true;
        return this.exec(gpgArgs, options);
      } else {
        throw err;
      }
    });
  }

  resolveDotGitDir() {
    var _this2 = this;

    return _asyncToGenerator(function* () {
      try {
        yield (0, _helpers.fsStat)(_this2.workingDir); // fails if folder doesn't exist
        const output = yield _this2.exec(['rev-parse', '--resolve-git-dir', _path2.default.join(_this2.workingDir, '.git')]);
        const dotGitDir = output.trim();
        if (_path2.default.isAbsolute(dotGitDir)) {
          return (0, _helpers.toNativePathSep)(dotGitDir);
        } else {
          return (0, _helpers.toNativePathSep)(_path2.default.resolve(_path2.default.join(_this2.workingDir, dotGitDir)));
        }
      } catch (e) {
        return null;
      }
    })();
  }

  init() {
    return this.exec(['init', this.workingDir]);
  }

  /**
   * Staging/Unstaging files and patches and committing
   */
  stageFiles(paths) {
    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['add'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  unstageFiles(paths) {
    let commit = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : 'HEAD';

    if (paths.length === 0) {
      return Promise.resolve(null);
    }
    const args = ['reset', commit, '--'].concat(paths.map(_helpers.toGitPathSep));
    return this.exec(args, { writeOperation: true });
  }

  applyPatch(patch) {
    var _ref8 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let index = _ref8.index;

    const args = ['apply', '-'];
    if (index) {
      args.splice(1, 0, '--cached');
    }
    return this.exec(args, { stdin: patch, writeOperation: true });
  }

  commit(message) {
    var _ref9 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let allowEmpty = _ref9.allowEmpty,
        amend = _ref9.amend;

    const args = ['commit', '-m', message];
    if (amend) {
      args.push('--amend');
    }
    if (allowEmpty) {
      args.push('--allow-empty');
    }
    return this.gpgExec(args, { writeOperation: true });
  }

  /**
   * File Status and Diffs
   */
  getStatusBundle() {
    var _this3 = this;

    return _asyncToGenerator(function* () {
      const args = ['status', '--porcelain=v2', '--branch', '--untracked-files=all', '--ignore-submodules=dirty', '-z'];
      const output = yield _this3.exec(args);
      const results = yield (0, _whatTheStatus.parse)(output);

      for (const entryType in results) {
        if (Array.isArray(results[entryType])) {
          _this3.updateNativePathSepForEntries(results[entryType]);
        }
      }

      return results;
    })();
  }

  updateNativePathSepForEntries(entries) {
    entries.forEach(entry => {
      // Normally we would avoid mutating responses from other package's APIs, but we control
      // the `what-the-status` module and know there are no side effects.
      // This is a hot code path and by mutating we avoid creating new objects that will just be GC'ed
      if (entry.filePath) {
        entry.filePath = (0, _helpers.toNativePathSep)(entry.filePath);
      }
      if (entry.origFilePath) {
        entry.origFilePath = (0, _helpers.toNativePathSep)(entry.origFilePath);
      }
    });
  }

  diffFileStatus() {
    var _this4 = this;

    let options = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
    return _asyncToGenerator(function* () {
      const args = ['diff', '--name-status', '--no-renames'];
      if (options.staged) {
        args.push('--staged');
      }
      if (options.target) {
        args.push(options.target);
      }
      const output = yield _this4.exec(args);

      const statusMap = {
        A: 'added',
        M: 'modified',
        D: 'deleted',
        U: 'unmerged'
      };

      const fileStatuses = {};
      output && output.trim().split(LINE_ENDING_REGEX).forEach(function (line) {
        var _line$split = line.split('\t'),
            _line$split2 = _slicedToArray(_line$split, 2);

        const status = _line$split2[0],
              rawFilePath = _line$split2[1];

        const filePath = (0, _helpers.toNativePathSep)(rawFilePath);
        fileStatuses[filePath] = statusMap[status];
      });
      if (!options.staged) {
        const untracked = yield _this4.getUntrackedFiles();
        untracked.forEach(function (filePath) {
          fileStatuses[filePath] = 'added';
        });
      }
      return fileStatuses;
    })();
  }

  getUntrackedFiles() {
    var _this5 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this5.exec(['ls-files', '--others', '--exclude-standard']);
      if (output.trim() === '') {
        return [];
      }
      return output.trim().split(LINE_ENDING_REGEX).map(_helpers.toNativePathSep);
    })();
  }

  getDiffForFilePath(filePath) {
    var _this6 = this;

    var _ref10 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let staged = _ref10.staged,
        baseCommit = _ref10.baseCommit;
    return _asyncToGenerator(function* () {
      let args = ['diff', '--no-prefix', '--no-renames', '--diff-filter=u'];
      if (staged) {
        args.push('--staged');
      }
      if (baseCommit) {
        args.push(baseCommit);
      }
      args = args.concat(['--', (0, _helpers.toGitPathSep)(filePath)]);
      const output = yield _this6.exec(args);

      let rawDiffs = [];
      if (output) {
        rawDiffs = (0, _whatTheDiff.parse)(output).filter(function (rawDiff) {
          return rawDiff.status !== 'unmerged';
        });

        for (let i = 0; i < rawDiffs.length; i++) {
          const rawDiff = rawDiffs[i];
          if (rawDiff.oldPath) {
            rawDiff.oldPath = (0, _helpers.toNativePathSep)(rawDiff.oldPath);
          }
          if (rawDiff.newPath) {
            rawDiff.newPath = (0, _helpers.toNativePathSep)(rawDiff.newPath);
          }
        }
      }

      if (!staged && (yield _this6.getUntrackedFiles()).includes(filePath)) {
        // add untracked file
        const absPath = _path2.default.join(_this6.workingDir, filePath);
        const executable = yield (0, _helpers.isFileExecutable)(absPath);
        const contents = yield (0, _helpers.readFile)(absPath);
        const binary = (0, _helpers.isBinary)(contents);
        rawDiffs.push(buildAddedFilePatch(filePath, binary ? null : contents, executable));
      }
      if (rawDiffs.length > 1) {
        throw new Error(`Expected 0 or 1 diffs for ${filePath} but got ${rawDiffs.length}`);
      }
      return rawDiffs[0];
    })();
  }

  /**
   * Miscellaneous getters
   */
  getCommit(ref) {
    var _this7 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this7.exec(['log', '--pretty=%H%x00%B%x00', '--no-abbrev-commit', '-1', ref]);

      var _output$split = output.split('\0'),
          _output$split2 = _slicedToArray(_output$split, 2);

      const sha = _output$split2[0],
            message = _output$split2[1];

      return { sha, message: message.trim(), unbornRef: false };
    })();
  }

  getHeadCommit() {
    var _this8 = this;

    return _asyncToGenerator(function* () {
      try {
        const commit = yield _this8.getCommit('HEAD');
        commit.unbornRef = false;
        return commit;
      } catch (e) {
        if (/unknown revision/.test(e.stdErr)) {
          return { sha: '', message: '', unbornRef: true };
        } else {
          throw e;
        }
      }
    })();
  }

  readFileFromIndex(filePath) {
    return this.exec(['show', `:${(0, _helpers.toGitPathSep)(filePath)}`]);
  }

  /**
   * Merge
   */
  merge(branchName) {
    return this.gpgExec(['merge', branchName], { writeOperation: true });
  }

  isMerging(dotGitDir) {
    return _asyncToGenerator(function* () {
      try {
        yield (0, _helpers.readFile)(_path2.default.join(dotGitDir, 'MERGE_HEAD'));
        return true;
      } catch (e) {
        return false;
      }
    })();
  }

  abortMerge() {
    return this.exec(['merge', '--abort'], { writeOperation: true });
  }

  checkoutSide(side, paths) {
    if (paths.length === 0) {
      return Promise.resolve();
    }

    return this.exec(['checkout', `--${side}`, ...paths.map(_helpers.toGitPathSep)]);
  }

  /**
   * Rebase
   */
  isRebasing(dotGitDir) {
    return _asyncToGenerator(function* () {
      const results = yield Promise.all([(0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-merge')), (0, _helpers.fileExists)(_path2.default.join(dotGitDir, 'rebase-apply'))]);
      return results.some(function (r) {
        return r;
      });
    })();
  }

  /**
   * Remote interactions
   */
  clone(remoteUrl) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const args = ['clone'];
    if (options.noLocal) {
      args.push('--no-local');
    }
    if (options.bare) {
      args.push('--bare');
    }
    if (options.recursive) {
      args.push('--recursive');
    }
    args.push(remoteUrl, this.workingDir);

    return this.exec(args, { writeOperation: true });
  }

  fetch(remoteName, branchName) {
    return this.exec(['fetch', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  pull(remoteName, branchName) {
    return this.gpgExec(['pull', remoteName, branchName], { useGitPromptServer: true, writeOperation: true });
  }

  push(remoteName, branchName) {
    let options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    const args = ['push', remoteName || 'origin', branchName];
    if (options.setUpstream) {
      args.push('--set-upstream');
    }
    if (options.force) {
      args.push('--force');
    }
    return this.exec(args, { useGitPromptServer: true, writeOperation: true });
  }

  /**
   * Branches
   */
  checkout(branchName) {
    let options = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    const args = ['checkout'];
    if (options.createNew) {
      args.push('-b');
    }
    return this.exec(args.concat(branchName), { writeOperation: true });
  }

  checkoutFiles(paths, revision) {
    if (paths.length === 0) {
      return null;
    }
    const args = ['checkout'];
    if (revision) {
      args.push(revision);
    }
    return this.exec(args.concat('--', paths.map(_helpers.toGitPathSep)), { writeOperation: true });
  }

  getBranches() {
    var _this9 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this9.exec(['for-each-ref', '--format=%(refname:short)', 'refs/heads/**']);
      return output.trim().split(LINE_ENDING_REGEX);
    })();
  }

  describeHead() {
    var _this10 = this;

    return _asyncToGenerator(function* () {
      return (yield _this10.exec(['describe', '--contains', '--all', '--always', 'HEAD'])).trim();
    })();
  }

  getConfig(option) {
    var _this11 = this;

    var _ref11 = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};

    let local = _ref11.local;
    return _asyncToGenerator(function* () {
      let output;
      try {
        let args = ['config'];
        if (local) {
          args.push('--local');
        }
        args = args.concat(option);
        output = yield _this11.exec(args);
      } catch (err) {
        if (err.code === 1) {
          // No matching config found
          return null;
        } else {
          throw err;
        }
      }

      return output.trim();
    })();
  }

  setConfig(option, value) {
    var _ref12 = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};

    let replaceAll = _ref12.replaceAll;

    let args = ['config'];
    if (replaceAll) {
      args.push('--replace-all');
    }
    args = args.concat(option, value);
    return this.exec(args, { writeOperation: true });
  }

  unsetConfig(option) {
    return this.exec(['config', '--unset', option], { writeOperation: true });
  }

  getRemotes() {
    var _this12 = this;

    return _asyncToGenerator(function* () {
      let output = yield _this12.getConfig(['--get-regexp', '^remote\\..*\\.url$'], { local: true });
      if (output) {
        output = output.trim();
        if (!output.length) {
          return [];
        }
        return output.split('\n').map(function (line) {
          const match = line.match(/^remote\.(.*)\.url (.*)$/);
          return {
            name: match[1],
            url: match[2]
          };
        });
      } else {
        return [];
      }
    })();
  }

  createBlob() {
    var _this13 = this;

    var _ref13 = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};

    let filePath = _ref13.filePath,
        stdin = _ref13.stdin;
    return _asyncToGenerator(function* () {
      let output;
      if (filePath) {
        try {
          output = (yield _this13.exec(['hash-object', '-w', filePath], { writeOperation: true })).trim();
        } catch (e) {
          if (e.stdErr && e.stdErr.match(/fatal: Cannot open .*: No such file or directory/)) {
            output = null;
          } else {
            throw e;
          }
        }
      } else if (stdin) {
        output = (yield _this13.exec(['hash-object', '-w', '--stdin'], { stdin, writeOperation: true })).trim();
      } else {
        throw new Error('Must supply file path or stdin');
      }
      return output;
    })();
  }

  expandBlobToFile(absFilePath, sha) {
    var _this14 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this14.exec(['cat-file', '-p', sha]);
      yield (0, _helpers.writeFile)(absFilePath, output);
      return absFilePath;
    })();
  }

  getBlobContents(sha) {
    var _this15 = this;

    return _asyncToGenerator(function* () {
      return yield _this15.exec(['cat-file', '-p', sha]);
    })();
  }

  mergeFile(oursPath, commonBasePath, theirsPath, resultPath) {
    var _this16 = this;

    return _asyncToGenerator(function* () {
      const args = ['merge-file', '-p', oursPath, commonBasePath, theirsPath, '-L', 'current', '-L', 'after discard', '-L', 'before discard'];
      let output;
      let conflict = false;
      try {
        output = yield _this16.exec(args);
      } catch (e) {
        if (e instanceof GitError && e.code === 1) {
          output = e.stdOut;
          conflict = true;
        } else {
          throw e;
        }
      }

      // Interpret a relative resultPath as relative to the repository working directory for consistency with the
      // other arguments.
      const resolvedResultPath = _path2.default.resolve(_this16.workingDir, resultPath);
      yield (0, _helpers.writeFile)(resolvedResultPath, output);

      return { filePath: oursPath, resultPath, conflict };
    })();
  }

  writeMergeConflictToIndex(filePath, commonBaseSha, oursSha, theirsSha) {
    var _this17 = this;

    return _asyncToGenerator(function* () {
      const gitFilePath = (0, _helpers.toGitPathSep)(filePath);
      const fileMode = yield _this17.getFileMode(filePath);
      let indexInfo = `0 0000000000000000000000000000000000000000\t${gitFilePath}\n`;
      if (commonBaseSha) {
        indexInfo += `${fileMode} ${commonBaseSha} 1\t${gitFilePath}\n`;
      }
      if (oursSha) {
        indexInfo += `${fileMode} ${oursSha} 2\t${gitFilePath}\n`;
      }
      if (theirsSha) {
        indexInfo += `${fileMode} ${theirsSha} 3\t${gitFilePath}\n`;
      }
      return _this17.exec(['update-index', '--index-info'], { stdin: indexInfo, writeOperation: true });
    })();
  }

  getFileMode(filePath) {
    var _this18 = this;

    return _asyncToGenerator(function* () {
      const output = yield _this18.exec(['ls-files', '--stage', '--', (0, _helpers.toGitPathSep)(filePath)]);
      if (output) {
        return output.slice(0, 6);
      } else {
        const executable = yield (0, _helpers.isFileExecutable)(_path2.default.join(_this18.workingDir, filePath));
        return executable ? '100755' : '100644';
      }
    })();
  }

  destroy() {
    this.commandQueue.dispose();
  }
}, _class.defaultExecArgs = { stdin: null, useGitPromptServer: false, writeOperation: false }, _temp);
exports.default = GitShellOutStrategy;


function buildAddedFilePatch(filePath, contents, executable) {
  const hunks = [];
  if (contents) {
    const noNewLine = contents[contents.length - 1] !== '\n';
    const lines = contents.trim().split(LINE_ENDING_REGEX).map(line => `+${line}`);
    if (noNewLine) {
      lines.push('\\ No newline at end of file');
    }
    hunks.push({
      lines,
      oldStartLine: 0,
      oldLineCount: 0,
      newStartLine: 1,
      heading: '',
      newLineCount: noNewLine ? lines.length - 1 : lines.length
    });
  }
  return {
    oldPath: null,
    newPath: (0, _helpers.toNativePathSep)(filePath),
    oldMode: null,
    newMode: executable ? '100755' : '100644',
    status: 'added',
    hunks
  };
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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