(function() {
  var Emitter, Result, ResultsModel, TextEditor, _, binaryIndex, escapeHelper, ref, stringCompare,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  _ = require('underscore-plus');

  ref = require('atom'), Emitter = ref.Emitter, TextEditor = ref.TextEditor;

  escapeHelper = require('../escape-helper');

  Result = (function() {
    Result.create = function(result) {
      var matches, ref1;
      if (result != null ? (ref1 = result.matches) != null ? ref1.length : void 0 : void 0) {
        matches = result.matches.map(function(m) {
          return {
            matchText: m.matchText,
            lineText: m.lineText,
            lineTextOffset: m.lineTextOffset,
            range: m.range,
            leadingContextLines: m.leadingContextLines,
            trailingContextLines: m.trailingContextLines
          };
        });
        return new Result({
          filePath: result.filePath,
          matches: matches
        });
      } else {
        return null;
      }
    };

    function Result(result) {
      _.extend(this, result);
    }

    return Result;

  })();

  module.exports = ResultsModel = (function() {
    function ResultsModel(findOptions) {
      this.findOptions = findOptions;
      this.onContentsModified = bind(this.onContentsModified, this);
      this.emitter = new Emitter;
      atom.workspace.getCenter().observeActivePaneItem((function(_this) {
        return function(item) {
          if (item instanceof TextEditor) {
            return item.onDidStopChanging(function() {
              return _this.onContentsModified(item);
            });
          }
        };
      })(this));
      this.clear();
    }

    ResultsModel.prototype.onDidClear = function(callback) {
      return this.emitter.on('did-clear', callback);
    };

    ResultsModel.prototype.onDidClearSearchState = function(callback) {
      return this.emitter.on('did-clear-search-state', callback);
    };

    ResultsModel.prototype.onDidClearReplacementState = function(callback) {
      return this.emitter.on('did-clear-replacement-state', callback);
    };

    ResultsModel.prototype.onDidSearchPaths = function(callback) {
      return this.emitter.on('did-search-paths', callback);
    };

    ResultsModel.prototype.onDidErrorForPath = function(callback) {
      return this.emitter.on('did-error-for-path', callback);
    };

    ResultsModel.prototype.onDidStartSearching = function(callback) {
      return this.emitter.on('did-start-searching', callback);
    };

    ResultsModel.prototype.onDidCancelSearching = function(callback) {
      return this.emitter.on('did-cancel-searching', callback);
    };

    ResultsModel.prototype.onDidFinishSearching = function(callback) {
      return this.emitter.on('did-finish-searching', callback);
    };

    ResultsModel.prototype.onDidStartReplacing = function(callback) {
      return this.emitter.on('did-start-replacing', callback);
    };

    ResultsModel.prototype.onDidFinishReplacing = function(callback) {
      return this.emitter.on('did-finish-replacing', callback);
    };

    ResultsModel.prototype.onDidSearchPath = function(callback) {
      return this.emitter.on('did-search-path', callback);
    };

    ResultsModel.prototype.onDidReplacePath = function(callback) {
      return this.emitter.on('did-replace-path', callback);
    };

    ResultsModel.prototype.onDidAddResult = function(callback) {
      return this.emitter.on('did-add-result', callback);
    };

    ResultsModel.prototype.onDidRemoveResult = function(callback) {
      return this.emitter.on('did-remove-result', callback);
    };

    ResultsModel.prototype.clear = function() {
      this.clearSearchState();
      this.clearReplacementState();
      return this.emitter.emit('did-clear', this.getResultsSummary());
    };

    ResultsModel.prototype.clearSearchState = function() {
      this.pathCount = 0;
      this.matchCount = 0;
      this.regex = null;
      this.results = {};
      this.paths = [];
      this.active = false;
      this.searchErrors = null;
      if (this.inProgressSearchPromise != null) {
        this.inProgressSearchPromise.cancel();
        this.inProgressSearchPromise = null;
      }
      return this.emitter.emit('did-clear-search-state', this.getResultsSummary());
    };

    ResultsModel.prototype.clearReplacementState = function() {
      this.replacePattern = null;
      this.replacedPathCount = null;
      this.replacementCount = null;
      this.replacementErrors = null;
      return this.emitter.emit('did-clear-replacement-state', this.getResultsSummary());
    };

    ResultsModel.prototype.shoudldRerunSearch = function(findPattern, pathsPattern, replacePattern, options) {
      var onlyRunIfChanged;
      if (options == null) {
        options = {};
      }
      onlyRunIfChanged = options.onlyRunIfChanged;
      if (onlyRunIfChanged && (findPattern != null) && (pathsPattern != null) && findPattern === this.lastFindPattern && pathsPattern === this.lastPathsPattern) {
        return false;
      } else {
        return true;
      }
    };

    ResultsModel.prototype.search = function(findPattern, pathsPattern, replacePattern, options) {
      var keepReplacementState, leadingContextLineCount, onPathsSearched, searchPaths, trailingContextLineCount;
      if (options == null) {
        options = {};
      }
      if (!this.shoudldRerunSearch(findPattern, pathsPattern, replacePattern, options)) {
        return Promise.resolve();
      }
      keepReplacementState = options.keepReplacementState;
      if (keepReplacementState) {
        this.clearSearchState();
      } else {
        this.clear();
      }
      this.lastFindPattern = findPattern;
      this.lastPathsPattern = pathsPattern;
      this.findOptions.set(_.extend({
        findPattern: findPattern,
        replacePattern: replacePattern,
        pathsPattern: pathsPattern
      }, options));
      this.regex = this.findOptions.getFindPatternRegex();
      this.active = true;
      searchPaths = this.pathsArrayFromPathsPattern(pathsPattern);
      onPathsSearched = (function(_this) {
        return function(numberOfPathsSearched) {
          return _this.emitter.emit('did-search-paths', numberOfPathsSearched);
        };
      })(this);
      leadingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountBefore');
      trailingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountAfter');
      this.inProgressSearchPromise = atom.workspace.scan(this.regex, {
        paths: searchPaths,
        onPathsSearched: onPathsSearched,
        leadingContextLineCount: leadingContextLineCount,
        trailingContextLineCount: trailingContextLineCount
      }, (function(_this) {
        return function(result, error) {
          if (result) {
            return _this.setResult(result.filePath, Result.create(result));
          } else {
            if (_this.searchErrors == null) {
              _this.searchErrors = [];
            }
            _this.searchErrors.push(error);
            return _this.emitter.emit('did-error-for-path', error);
          }
        };
      })(this));
      this.emitter.emit('did-start-searching', this.inProgressSearchPromise);
      return this.inProgressSearchPromise.then((function(_this) {
        return function(message) {
          if (message === 'cancelled') {
            return _this.emitter.emit('did-cancel-searching');
          } else {
            _this.inProgressSearchPromise = null;
            return _this.emitter.emit('did-finish-searching', _this.getResultsSummary());
          }
        };
      })(this));
    };

    ResultsModel.prototype.replace = function(pathsPattern, replacePattern, replacementPaths) {
      var promise;
      if (!(this.findOptions.findPattern && (this.regex != null))) {
        return;
      }
      this.findOptions.set({
        replacePattern: replacePattern,
        pathsPattern: pathsPattern
      });
      if (this.findOptions.useRegex) {
        replacePattern = escapeHelper.unescapeEscapeSequence(replacePattern);
      }
      this.active = false;
      this.replacedPathCount = 0;
      this.replacementCount = 0;
      promise = atom.workspace.replace(this.regex, replacePattern, replacementPaths, (function(_this) {
        return function(result, error) {
          if (result) {
            if (result.replacements) {
              _this.replacedPathCount++;
              _this.replacementCount += result.replacements;
            }
            return _this.emitter.emit('did-replace-path', result);
          } else {
            if (_this.replacementErrors == null) {
              _this.replacementErrors = [];
            }
            _this.replacementErrors.push(error);
            return _this.emitter.emit('did-error-for-path', error);
          }
        };
      })(this));
      this.emitter.emit('did-start-replacing', promise);
      return promise.then((function(_this) {
        return function() {
          _this.emitter.emit('did-finish-replacing', _this.getResultsSummary());
          return _this.search(_this.findOptions.findPattern, _this.findOptions.pathsPattern, _this.findOptions.replacePattern, {
            keepReplacementState: true
          });
        };
      })(this))["catch"](function(e) {
        return console.error(e.stack);
      });
    };

    ResultsModel.prototype.setActive = function(isActive) {
      if ((isActive && this.findOptions.findPattern) || !isActive) {
        return this.active = isActive;
      }
    };

    ResultsModel.prototype.getActive = function() {
      return this.active;
    };

    ResultsModel.prototype.getFindOptions = function() {
      return this.findOptions;
    };

    ResultsModel.prototype.getLastFindPattern = function() {
      return this.lastFindPattern;
    };

    ResultsModel.prototype.getResultsSummary = function() {
      var findPattern, ref1, replacePattern;
      findPattern = (ref1 = this.lastFindPattern) != null ? ref1 : this.findOptions.findPattern;
      replacePattern = this.findOptions.replacePattern;
      return {
        findPattern: findPattern,
        replacePattern: replacePattern,
        pathCount: this.pathCount,
        matchCount: this.matchCount,
        searchErrors: this.searchErrors,
        replacedPathCount: this.replacedPathCount,
        replacementCount: this.replacementCount,
        replacementErrors: this.replacementErrors
      };
    };

    ResultsModel.prototype.getPathCount = function() {
      return this.pathCount;
    };

    ResultsModel.prototype.getMatchCount = function() {
      return this.matchCount;
    };

    ResultsModel.prototype.getPaths = function() {
      return this.paths;
    };

    ResultsModel.prototype.getResult = function(filePath) {
      return this.results[filePath];
    };

    ResultsModel.prototype.getResultAt = function(index) {
      return this.results[this.paths[index]];
    };

    ResultsModel.prototype.setResult = function(filePath, result) {
      if (result) {
        return this.addResult(filePath, result);
      } else {
        return this.removeResult(filePath);
      }
    };

    ResultsModel.prototype.addResult = function(filePath, result) {
      var filePathInsertedIndex;
      filePathInsertedIndex = null;
      if (this.results[filePath]) {
        this.matchCount -= this.results[filePath].matches.length;
      } else {
        this.pathCount++;
        filePathInsertedIndex = binaryIndex(this.paths, filePath, stringCompare);
        this.paths.splice(filePathInsertedIndex, 0, filePath);
      }
      this.matchCount += result.matches.length;
      this.results[filePath] = result;
      return this.emitter.emit('did-add-result', {
        filePath: filePath,
        result: result,
        filePathInsertedIndex: filePathInsertedIndex
      });
    };

    ResultsModel.prototype.removeResult = function(filePath) {
      if (this.results[filePath]) {
        this.pathCount--;
        this.matchCount -= this.results[filePath].matches.length;
        this.paths = _.without(this.paths, filePath);
        delete this.results[filePath];
        return this.emitter.emit('did-remove-result', {
          filePath: filePath
        });
      }
    };

    ResultsModel.prototype.onContentsModified = function(editor) {
      var leadingContextLineCount, matches, result, trailingContextLineCount;
      if (!(this.active && this.regex)) {
        return;
      }
      if (!editor.getPath()) {
        return;
      }
      matches = [];
      if (parseFloat(atom.getVersion()) >= 1.17) {
        leadingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountBefore');
        trailingContextLineCount = atom.config.get('find-and-replace.searchContextLineCountAfter');
        editor.scan(this.regex, {
          leadingContextLineCount: leadingContextLineCount,
          trailingContextLineCount: trailingContextLineCount
        }, function(match) {
          return matches.push(match);
        });
      } else {
        editor.scan(this.regex, function(match) {
          return matches.push(match);
        });
      }
      result = Result.create({
        filePath: editor.getPath(),
        matches: matches
      });
      this.setResult(editor.getPath(), result);
      return this.emitter.emit('did-finish-searching', this.getResultsSummary());
    };

    ResultsModel.prototype.pathsArrayFromPathsPattern = function(pathsPattern) {
      var i, inputPath, len, ref1, results;
      ref1 = pathsPattern.trim().split(',');
      results = [];
      for (i = 0, len = ref1.length; i < len; i++) {
        inputPath = ref1[i];
        if (inputPath) {
          results.push(inputPath.trim());
        }
      }
      return results;
    };

    return ResultsModel;

  })();

  stringCompare = function(a, b) {
    return a.localeCompare(b);
  };

  binaryIndex = function(array, value, comparator) {
    var high, low, mid;
    low = 0;
    high = array.length;
    while (low < high) {
      mid = Math.floor((low + high) / 2);
      if (comparator(array[mid], value) < 0) {
        low = mid + 1;
      } else {
        high = mid;
      }
    }
    return low;
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
