(function() {
  var BracketMatcherView, CompositeDisposable, MAX_ROWS_TO_SCAN, MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL, MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL, ONE_CHAR_BACKWARD_TRAVERSAL, ONE_CHAR_FORWARD_TRAVERSAL, Point, Range, TWO_CHARS_BACKWARD_TRAVERSAL, TagFinder, _, ref,
    bind = function(fn, me){ return function(){ return fn.apply(me, arguments); }; };

  CompositeDisposable = require('atom').CompositeDisposable;

  _ = require('underscore-plus');

  ref = require('atom'), Range = ref.Range, Point = ref.Point;

  TagFinder = require('./tag-finder');

  MAX_ROWS_TO_SCAN = 10000;

  ONE_CHAR_FORWARD_TRAVERSAL = Object.freeze(Point(0, 1));

  ONE_CHAR_BACKWARD_TRAVERSAL = Object.freeze(Point(0, -1));

  TWO_CHARS_BACKWARD_TRAVERSAL = Object.freeze(Point(0, -2));

  MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL = Object.freeze(Point(MAX_ROWS_TO_SCAN, 0));

  MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL = Object.freeze(Point(-MAX_ROWS_TO_SCAN, 0));

  module.exports = BracketMatcherView = (function() {
    function BracketMatcherView(editor, editorElement, matchManager) {
      this.editor = editor;
      this.matchManager = matchManager;
      this.updateMatch = bind(this.updateMatch, this);
      this.destroy = bind(this.destroy, this);
      this.subscriptions = new CompositeDisposable;
      this.tagFinder = new TagFinder(this.editor);
      this.pairHighlighted = false;
      this.tagHighlighted = false;
      this.subscriptions.add(this.editor.onDidTokenize(this.updateMatch));
      this.subscriptions.add(this.editor.getBuffer().onDidChangeText(this.updateMatch));
      this.subscriptions.add(this.editor.onDidChangeGrammar(this.updateMatch));
      this.subscriptions.add(this.editor.onDidChangeSelectionRange(this.updateMatch));
      this.subscriptions.add(this.editor.onDidAddCursor(this.updateMatch));
      this.subscriptions.add(this.editor.onDidRemoveCursor(this.updateMatch));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:go-to-matching-bracket', (function(_this) {
        return function() {
          return _this.goToMatchingPair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:go-to-enclosing-bracket', (function(_this) {
        return function() {
          return _this.goToEnclosingPair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:select-inside-brackets', (function(_this) {
        return function() {
          return _this.selectInsidePair();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:close-tag', (function(_this) {
        return function() {
          return _this.closeTag();
        };
      })(this)));
      this.subscriptions.add(atom.commands.add(editorElement, 'bracket-matcher:remove-matching-brackets', (function(_this) {
        return function() {
          return _this.removeMatchingBrackets();
        };
      })(this)));
      this.subscriptions.add(this.editor.onDidDestroy(this.destroy));
      this.updateMatch();
    }

    BracketMatcherView.prototype.destroy = function() {
      return this.subscriptions.dispose();
    };

    BracketMatcherView.prototype.updateMatch = function() {
      var currentPair, endRange, highlightPair, highlightTag, matchPosition, matchingPair, pair, position, ref1, ref2, startRange;
      if (this.pairHighlighted) {
        this.editor.destroyMarker(this.startMarker.id);
        this.editor.destroyMarker(this.endMarker.id);
      }
      this.pairHighlighted = false;
      this.tagHighlighted = false;
      if (!this.editor.getLastSelection().isEmpty()) {
        return;
      }
      ref1 = this.findCurrentPair(false), position = ref1.position, currentPair = ref1.currentPair, matchingPair = ref1.matchingPair;
      if (position) {
        matchPosition = this.findMatchingEndPair(position, currentPair, matchingPair);
      } else {
        ref2 = this.findCurrentPair(true), position = ref2.position, currentPair = ref2.currentPair, matchingPair = ref2.matchingPair;
        if (position) {
          matchPosition = this.findMatchingStartPair(position, matchingPair, currentPair);
        }
      }
      startRange = null;
      endRange = null;
      highlightTag = false;
      highlightPair = false;
      if ((position != null) && (matchPosition != null)) {
        startRange = Range(position, position.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        endRange = Range(matchPosition, matchPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        highlightPair = true;
      } else {
        if (pair = this.tagFinder.findMatchingTags()) {
          startRange = pair.startRange;
          endRange = pair.endRange;
          highlightPair = true;
          highlightTag = true;
        }
      }
      if (!(highlightTag || highlightPair)) {
        return;
      }
      if (this.editor.isFoldedAtCursorRow()) {
        return;
      }
      if (this.isCursorOnCommentOrString()) {
        return;
      }
      this.startMarker = this.createMarker(startRange);
      this.endMarker = this.createMarker(endRange);
      this.pairHighlighted = highlightPair;
      return this.tagHighlighted = highlightTag;
    };

    BracketMatcherView.prototype.removeMatchingBrackets = function() {
      if (this.editor.hasMultipleCursors()) {
        return this.editor.backspace();
      }
      return this.editor.transact((function(_this) {
        return function() {
          var currentPair, matchPosition, matchingPair, position, ref1, ref2, text;
          if (_this.editor.getLastSelection().isEmpty()) {
            _this.editor.selectLeft();
          }
          text = _this.editor.getSelectedText();
          _this.editor.moveRight();
          if (_this.matchManager.pairedCharacters.hasOwnProperty(text) || _this.matchManager.pairedCharactersInverse.hasOwnProperty(text)) {
            ref1 = _this.findCurrentPair(false), position = ref1.position, currentPair = ref1.currentPair, matchingPair = ref1.matchingPair;
            if (position) {
              matchPosition = _this.findMatchingEndPair(position, currentPair, matchingPair);
            } else {
              ref2 = _this.findCurrentPair(true), position = ref2.position, currentPair = ref2.currentPair, matchingPair = ref2.matchingPair;
              if (position) {
                matchPosition = _this.findMatchingStartPair(position, matchingPair, currentPair);
              }
            }
            if ((position != null) && (matchPosition != null)) {
              _this.editor.setCursorBufferPosition(matchPosition);
              _this.editor["delete"]();
              if (position.row === matchPosition.row && _this.matchManager.pairedCharactersInverse.hasOwnProperty(currentPair)) {
                position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
              }
              _this.editor.setCursorBufferPosition(position);
              return _this.editor["delete"]();
            } else {
              return _this.editor.backspace();
            }
          } else {
            return _this.editor.backspace();
          }
        };
      })(this));
    };

    BracketMatcherView.prototype.findMatchingEndPair = function(startPairPosition, startPair, endPair) {
      var endPairPosition, scanRange, unpairedCount;
      if (startPair === endPair) {
        return;
      }
      scanRange = new Range(startPairPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL), startPairPosition.traverse(MAX_ROWS_TO_SCAN_FORWARD_TRAVERSAL));
      endPairPosition = null;
      unpairedCount = 0;
      this.editor.scanInBufferRange(this.matchManager.pairRegexes[startPair], scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          switch (result.match[0]) {
            case startPair:
              return unpairedCount++;
            case endPair:
              unpairedCount--;
              if (unpairedCount < 0) {
                endPairPosition = result.range.start;
                return result.stop();
              }
          }
        };
      })(this));
      return endPairPosition;
    };

    BracketMatcherView.prototype.findMatchingStartPair = function(endPairPosition, startPair, endPair) {
      var scanRange, startPairPosition, unpairedCount;
      if (startPair === endPair) {
        return;
      }
      scanRange = new Range(endPairPosition.traverse(MAX_ROWS_TO_SCAN_BACKWARD_TRAVERSAL), endPairPosition);
      startPairPosition = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(this.matchManager.pairRegexes[startPair], scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          switch (result.match[0]) {
            case startPair:
              unpairedCount--;
              if (unpairedCount < 0) {
                startPairPosition = result.range.start;
                return result.stop();
              }
              break;
            case endPair:
              return unpairedCount++;
          }
        };
      })(this));
      return startPairPosition;
    };

    BracketMatcherView.prototype.findAnyStartPair = function(cursorPosition) {
      var combinedRegExp, endPair, endPairRegExp, scanRange, startPair, startPairRegExp, startPosition, unpairedCount;
      scanRange = new Range(Point.ZERO, cursorPosition);
      startPair = _.escapeRegExp(_.keys(this.matchManager.pairedCharacters).join(''));
      endPair = _.escapeRegExp(_.keys(this.matchManager.pairedCharactersInverse).join(''));
      combinedRegExp = new RegExp("[" + startPair + endPair + "]", 'g');
      startPairRegExp = new RegExp("[" + startPair + "]", 'g');
      endPairRegExp = new RegExp("[" + endPair + "]", 'g');
      startPosition = null;
      unpairedCount = 0;
      this.editor.backwardsScanInBufferRange(combinedRegExp, scanRange, (function(_this) {
        return function(result) {
          if (_this.isRangeCommentedOrString(result.range)) {
            return;
          }
          if (result.match[0].match(endPairRegExp)) {
            return unpairedCount++;
          } else if (result.match[0].match(startPairRegExp)) {
            unpairedCount--;
            if (unpairedCount < 0) {
              startPosition = result.range.start;
              return result.stop();
            }
          }
        };
      })(this));
      return startPosition;
    };

    BracketMatcherView.prototype.createMarker = function(bufferRange) {
      var marker;
      marker = this.editor.markBufferRange(bufferRange);
      this.editor.decorateMarker(marker, {
        type: 'highlight',
        "class": 'bracket-matcher',
        deprecatedRegionClass: 'bracket-matcher'
      });
      return marker;
    };

    BracketMatcherView.prototype.findCurrentPair = function(isInverse) {
      var currentPair, matches, matchingPair, position;
      position = this.editor.getCursorBufferPosition();
      if (isInverse) {
        matches = this.matchManager.pairedCharactersInverse;
        position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
      } else {
        matches = this.matchManager.pairedCharacters;
      }
      currentPair = this.editor.getTextInRange(Range.fromPointWithDelta(position, 0, 1));
      if (!matches[currentPair]) {
        if (isInverse) {
          position = position.traverse(ONE_CHAR_FORWARD_TRAVERSAL);
        } else {
          position = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
        }
        currentPair = this.editor.getTextInRange(Range.fromPointWithDelta(position, 0, 1));
      }
      if (matchingPair = matches[currentPair]) {
        return {
          position: position,
          currentPair: currentPair,
          matchingPair: matchingPair
        };
      } else {
        return {};
      }
    };

    BracketMatcherView.prototype.goToMatchingPair = function() {
      var endPosition, endRange, position, previousPosition, ref1, startPosition, startRange, tagCharacterOffset, tagLength;
      if (!this.pairHighlighted) {
        return this.goToEnclosingPair();
      }
      position = this.editor.getCursorBufferPosition();
      if (this.tagHighlighted) {
        startRange = this.startMarker.getBufferRange();
        tagLength = startRange.end.column - startRange.start.column;
        endRange = this.endMarker.getBufferRange();
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        startRange = new Range(startRange.start.traverse(ONE_CHAR_BACKWARD_TRAVERSAL), endRange.end.traverse(ONE_CHAR_BACKWARD_TRAVERSAL));
        endRange = new Range(endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL), endRange.end.traverse(TWO_CHARS_BACKWARD_TRAVERSAL));
        if (position.isLessThan(endRange.start)) {
          tagCharacterOffset = position.column - startRange.start.column;
          if (tagCharacterOffset > 0) {
            tagCharacterOffset++;
          }
          tagCharacterOffset = Math.min(tagCharacterOffset, tagLength + 2);
          return this.editor.setCursorBufferPosition(endRange.start.traverse([0, tagCharacterOffset]));
        } else {
          tagCharacterOffset = position.column - endRange.start.column;
          if (tagCharacterOffset > 1) {
            tagCharacterOffset--;
          }
          tagCharacterOffset = Math.min(tagCharacterOffset, tagLength + 1);
          return this.editor.setCursorBufferPosition(startRange.start.traverse([0, tagCharacterOffset]));
        }
      } else {
        previousPosition = position.traverse(ONE_CHAR_BACKWARD_TRAVERSAL);
        startPosition = this.startMarker.getStartBufferPosition();
        endPosition = this.endMarker.getStartBufferPosition();
        if (position.isEqual(startPosition)) {
          return this.editor.setCursorBufferPosition(endPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        } else if (previousPosition.isEqual(startPosition)) {
          return this.editor.setCursorBufferPosition(endPosition);
        } else if (position.isEqual(endPosition)) {
          return this.editor.setCursorBufferPosition(startPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL));
        } else if (previousPosition.isEqual(endPosition)) {
          return this.editor.setCursorBufferPosition(startPosition);
        }
      }
    };

    BracketMatcherView.prototype.goToEnclosingPair = function() {
      var endRange, matchPosition, pair, ref1, startRange;
      if (this.pairHighlighted) {
        return;
      }
      if (matchPosition = this.findAnyStartPair(this.editor.getCursorBufferPosition())) {
        return this.editor.setCursorBufferPosition(matchPosition);
      } else if (pair = this.tagFinder.findEnclosingTags()) {
        startRange = pair.startRange, endRange = pair.endRange;
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        return this.editor.setCursorBufferPosition(pair.startRange.start);
      }
    };

    BracketMatcherView.prototype.selectInsidePair = function() {
      var endPosition, endRange, pair, rangeToSelect, ref1, ref2, startPair, startPosition, startRange;
      if (this.pairHighlighted) {
        startRange = this.startMarker.getBufferRange();
        endRange = this.endMarker.getBufferRange();
        if (startRange.compare(endRange) > 0) {
          ref1 = [endRange, startRange], startRange = ref1[0], endRange = ref1[1];
        }
        if (this.tagHighlighted) {
          startPosition = startRange.end;
          endPosition = endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL);
        } else {
          startPosition = startRange.start;
          endPosition = endRange.start;
        }
      } else {
        if (startPosition = this.findAnyStartPair(this.editor.getCursorBufferPosition())) {
          startPair = this.editor.getTextInRange(Range.fromPointWithDelta(startPosition, 0, 1));
          endPosition = this.findMatchingEndPair(startPosition, startPair, this.matchManager.pairedCharacters[startPair]);
        } else if (pair = this.tagFinder.findEnclosingTags()) {
          startRange = pair.startRange, endRange = pair.endRange;
          if (startRange.compare(endRange) > 0) {
            ref2 = [endRange, startRange], startRange = ref2[0], endRange = ref2[1];
          }
          startPosition = startRange.end;
          endPosition = endRange.start.traverse(TWO_CHARS_BACKWARD_TRAVERSAL);
        }
      }
      if ((startPosition != null) && (endPosition != null)) {
        rangeToSelect = new Range(startPosition.traverse(ONE_CHAR_FORWARD_TRAVERSAL), endPosition);
        return this.editor.setSelectedBufferRange(rangeToSelect);
      }
    };

    BracketMatcherView.prototype.closeTag = function() {
      var cursorPosition, postFragment, preFragment, tag;
      cursorPosition = this.editor.getCursorBufferPosition();
      preFragment = this.editor.getTextInBufferRange([Point.ZERO, cursorPosition]);
      postFragment = this.editor.getTextInBufferRange([cursorPosition, Point.INFINITY]);
      if (tag = this.tagFinder.closingTagForFragments(preFragment, postFragment)) {
        return this.editor.insertText("</" + tag + ">");
      }
    };

    BracketMatcherView.prototype.isCursorOnCommentOrString = function() {
      return this.isScopeCommentedOrString(this.editor.getLastCursor().getScopeDescriptor().getScopesArray());
    };

    BracketMatcherView.prototype.isRangeCommentedOrString = function(range) {
      return this.isScopeCommentedOrString(this.editor.scopeDescriptorForBufferPosition(range.start).getScopesArray());
    };

    BracketMatcherView.prototype.isScopeCommentedOrString = function(scopesArray) {
      var i, len, ref1, scope;
      ref1 = scopesArray.reverse();
      for (i = 0, len = ref1.length; i < len; i++) {
        scope = ref1[i];
        scope = scope.split('.');
        if (scope.includes('embedded') && scope.includes('source')) {
          return false;
        }
        if (scope.includes('comment') || scope.includes('string')) {
          return true;
        }
      }
      return false;
    };

    return BracketMatcherView;

  })();

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
