(function() {
  var CharacterPattern, _;

  _ = require('underscore-plus');

  CharacterPattern = /[\w\u0410-\u042F\u0401\u0430-\u044F\u0451]/;

  module.exports = {
    activate: function() {
      return atom.commands.add('atom-text-editor', {
        'autoflow:reflow-selection': (function(_this) {
          return function(event) {
            return _this.reflowSelection(event.currentTarget.getModel());
          };
        })(this)
      });
    },
    reflowSelection: function(editor) {
      var range, reflowOptions, reflowedText;
      range = editor.getSelectedBufferRange();
      if (range.isEmpty()) {
        range = editor.getCurrentParagraphBufferRange();
      }
      if (range == null) {
        return;
      }
      reflowOptions = {
        wrapColumn: this.getPreferredLineLength(editor),
        tabLength: this.getTabLength(editor)
      };
      reflowedText = this.reflow(editor.getTextInRange(range), reflowOptions);
      return editor.getBuffer().setTextInRange(range, reflowedText);
    },
    reflow: function(text, arg) {
      var block, blockLines, currentLine, currentLineLength, escapedLinePrefix, i, j, leadingVerticalSpace, len, len1, linePrefix, linePrefixTabExpanded, lines, paragraphBlocks, paragraphs, ref, segment, tabLength, tabLengthInSpaces, trailingVerticalSpace, wrapColumn;
      wrapColumn = arg.wrapColumn, tabLength = arg.tabLength;
      paragraphs = [];
      text = text.replace(/\r\n?/g, '\n');
      leadingVerticalSpace = text.match(/^\s*\n/);
      if (leadingVerticalSpace) {
        text = text.substr(leadingVerticalSpace.length);
      } else {
        leadingVerticalSpace = '';
      }
      trailingVerticalSpace = text.match(/\n\s*$/);
      if (trailingVerticalSpace) {
        text = text.substr(0, text.length - trailingVerticalSpace.length);
      } else {
        trailingVerticalSpace = '';
      }
      paragraphBlocks = text.split(/\n\s*\n/g);
      if (tabLength) {
        tabLengthInSpaces = Array(tabLength + 1).join(' ');
      } else {
        tabLengthInSpaces = '';
      }
      for (i = 0, len = paragraphBlocks.length; i < len; i++) {
        block = paragraphBlocks[i];
        linePrefix = block.match(/^\s*[\/#*-]*\s*/g)[0];
        linePrefixTabExpanded = linePrefix;
        if (tabLengthInSpaces) {
          linePrefixTabExpanded = linePrefix.replace(/\t/g, tabLengthInSpaces);
        }
        blockLines = block.split('\n');
        if (linePrefix) {
          escapedLinePrefix = _.escapeRegExp(linePrefix);
          blockLines = blockLines.map(function(blockLine) {
            return blockLine.replace(RegExp("^" + escapedLinePrefix), '');
          });
        }
        blockLines = blockLines.map(function(blockLine) {
          return blockLine.replace(/^\s+/, '');
        });
        lines = [];
        currentLine = [];
        currentLineLength = linePrefixTabExpanded.length;
        ref = this.segmentText(blockLines.join(' '));
        for (j = 0, len1 = ref.length; j < len1; j++) {
          segment = ref[j];
          if (this.wrapSegment(segment, currentLineLength, wrapColumn)) {
            lines.push(linePrefix + currentLine.join(''));
            currentLine = [];
            currentLineLength = linePrefixTabExpanded.length;
          }
          currentLine.push(segment);
          currentLineLength += segment.length;
        }
        lines.push(linePrefix + currentLine.join(''));
        paragraphs.push(lines.join('\n').replace(/\s+\n/g, '\n'));
      }
      return leadingVerticalSpace + paragraphs.join('\n\n') + trailingVerticalSpace;
    },
    getTabLength: function(editor) {
      var ref;
      return (ref = atom.config.get('editor.tabLength', {
        scope: editor.getRootScopeDescriptor()
      })) != null ? ref : 2;
    },
    getPreferredLineLength: function(editor) {
      return atom.config.get('editor.preferredLineLength', {
        scope: editor.getRootScopeDescriptor()
      });
    },
    wrapSegment: function(segment, currentLineLength, wrapColumn) {
      return CharacterPattern.test(segment) && (currentLineLength + segment.length > wrapColumn) && (currentLineLength > 0 || segment.length < wrapColumn);
    },
    segmentText: function(text) {
      var match, re, segments;
      segments = [];
      re = /[\s]+|[^\s]+/g;
      while (match = re.exec(text)) {
        segments.push(match[0]);
      }
      return segments;
    }
  };

}).call(this);

//# sourceMappingURL=data:application/json;base64,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
