/*$
mlk in apdtool
Copyright (c) 2020 Azel

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
$*/

/*****************************************
 * mSaveImage
 *****************************************/

#include <string.h>
#include <stdio.h>

#include "mlk.h"
#include "mlk_saveimage.h"
#include "mlk_stdio.h"
#include "mlk_util.h"



/**@ mSaveImage 構造体を初期化
 *
 * @d:ゼロクリアする。 */

void mSaveImage_init(mSaveImage *p)
{
	memset(p, 0, sizeof(mSaveImage));
}


//===========================
// 内部で使われるもの
//===========================


/**@ ファイルを開く
 *
 * @r:FILE *。NULL で失敗。 */

void *mSaveImage_openFile(mSaveImage *p)
{
	switch(p->open.type)
	{
		case MSAVEIMAGE_OPEN_FILENAME:
			return mFILEopen(p->open.filename, "wb");
		case MSAVEIMAGE_OPEN_FP:
			return p->open.fp;
	}

	return NULL;
}

/**@ ファイルを閉じる */

void mSaveImage_closeFile(mSaveImage *p,void *fp)
{
	if(!fp) return;
	
	switch(p->open.type)
	{
		case MSAVEIMAGE_OPEN_FILENAME:
			fclose((FILE *)fp);
			break;
		case MSAVEIMAGE_OPEN_FP:
			fflush((FILE *)fp);
			break;
	}
}

/**@ 解像度を dpi 単位で取得
 *
 * @p:horz,vert 取得できなかった場合、0 にセットされる。
 * @r:FALSE で dpi で取得できなかった */

mlkbool mSaveImage_getDPI(mSaveImage *p,int *horz,int *vert)
{
	int h,v;

	h = p->reso_horz;
	v = p->reso_vert;
	
	switch(p->reso_unit)
	{
		//DPI
		case MSAVEIMAGE_RESOUNIT_DPI:
			*horz = h;
			*vert = v;
			return TRUE;
		//DPM -> DPI
		case MSAVEIMAGE_RESOUNIT_DPM:
			*horz = mDPMtoDPI(h);
			*vert = mDPMtoDPI(v);
			return TRUE;
	}

	*horz = 0;
	*vert = 0;

	return FALSE;
}

/**@ 解像度を dpm 単位で取得
 *
 * @p:horz,vert 取得できなかった場合、0 にセットされる。
 * @r:FALSE で dpm で取得できなかった */

mlkbool mSaveImage_getDPM(mSaveImage *p,int *horz,int *vert)
{
	int h,v;

	h = p->reso_horz;
	v = p->reso_vert;
	
	switch(p->reso_unit)
	{
		//DPI -> DPM
		case MSAVEIMAGE_RESOUNIT_DPI:
			*horz = mDPItoDPM(h);
			*vert = mDPItoDPM(v);
			return TRUE;
		//DPM
		case MSAVEIMAGE_RESOUNIT_DPM:
			*horz = h;
			*vert = v;
			return TRUE;
	}

	*horz = 0;
	*vert = 0;

	return FALSE;
}

/**@ パレットデータを RGBX → RGB (3byte) に変換して作成
 *
 * @d:palette_buf のパレットデータを変換し、新たにバッファを確保してセットする。
 * @r:確保されたバッファ */

uint8_t *mSaveImage_createPaletteRGB(mSaveImage *p)
{
	uint8_t *buf,*ps,*pd;
	int num,i;

	if(!p->palette_buf) return NULL;

	num = p->palette_num;

	buf = (uint8_t *)mMalloc(num * 3);
	if(!buf) return NULL;

	//RGBX => RGB 変換

	ps = p->palette_buf;
	pd = buf;

	for(i = num; i > 0; i--, ps += 4, pd += 3)
	{
		pd[0] = ps[0];
		pd[1] = ps[1];
		pd[2] = ps[2];
	}

	return buf;
}

