﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSStreamFwd.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ec2/EC2_EXPORTS.h>
#include <aws/ec2/model/ImageReferenceResourceType.h>
#include <aws/ec2/model/ResourceTypeOption.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Xml {
class XmlNode;
}  // namespace Xml
}  // namespace Utils
namespace EC2 {
namespace Model {

/**
 * <p>A resource type to check for image references. Associated options can also be
 * specified if the resource type is an EC2 instance or launch
 * template.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ec2-2016-11-15/ResourceTypeRequest">AWS
 * API Reference</a></p>
 */
class ResourceTypeRequest {
 public:
  AWS_EC2_API ResourceTypeRequest() = default;
  AWS_EC2_API ResourceTypeRequest(const Aws::Utils::Xml::XmlNode& xmlNode);
  AWS_EC2_API ResourceTypeRequest& operator=(const Aws::Utils::Xml::XmlNode& xmlNode);

  AWS_EC2_API void OutputToStream(Aws::OStream& ostream, const char* location, unsigned index, const char* locationValue) const;
  AWS_EC2_API void OutputToStream(Aws::OStream& oStream, const char* location) const;

  ///@{
  /**
   * <p>The resource type.</p>
   */
  inline ImageReferenceResourceType GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  inline void SetResourceType(ImageReferenceResourceType value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = value;
  }
  inline ResourceTypeRequest& WithResourceType(ImageReferenceResourceType value) {
    SetResourceType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The options that affect the scope of the response. Valid only when
   * <code>ResourceType</code> is <code>ec2:Instance</code> or
   * <code>ec2:LaunchTemplate</code>.</p>
   */
  inline const Aws::Vector<ResourceTypeOption>& GetResourceTypeOptions() const { return m_resourceTypeOptions; }
  inline bool ResourceTypeOptionsHasBeenSet() const { return m_resourceTypeOptionsHasBeenSet; }
  template <typename ResourceTypeOptionsT = Aws::Vector<ResourceTypeOption>>
  void SetResourceTypeOptions(ResourceTypeOptionsT&& value) {
    m_resourceTypeOptionsHasBeenSet = true;
    m_resourceTypeOptions = std::forward<ResourceTypeOptionsT>(value);
  }
  template <typename ResourceTypeOptionsT = Aws::Vector<ResourceTypeOption>>
  ResourceTypeRequest& WithResourceTypeOptions(ResourceTypeOptionsT&& value) {
    SetResourceTypeOptions(std::forward<ResourceTypeOptionsT>(value));
    return *this;
  }
  template <typename ResourceTypeOptionsT = ResourceTypeOption>
  ResourceTypeRequest& AddResourceTypeOptions(ResourceTypeOptionsT&& value) {
    m_resourceTypeOptionsHasBeenSet = true;
    m_resourceTypeOptions.emplace_back(std::forward<ResourceTypeOptionsT>(value));
    return *this;
  }
  ///@}
 private:
  ImageReferenceResourceType m_resourceType{ImageReferenceResourceType::NOT_SET};

  Aws::Vector<ResourceTypeOption> m_resourceTypeOptions;
  bool m_resourceTypeHasBeenSet = false;
  bool m_resourceTypeOptionsHasBeenSet = false;
};

}  // namespace Model
}  // namespace EC2
}  // namespace Aws
