"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.awsKmsMrkAreUnique = void 0;
const arn_parsing_1 = require("./arn_parsing");
//= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
//# The caller MUST provide:
function awsKmsMrkAreUnique(awsKmsIdentifers) {
    const multiRegionKeys = awsKmsIdentifers.filter((i) => (0, arn_parsing_1.isMultiRegionAwsKmsIdentifier)(i));
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If the list does not contain any multi-Region keys (aws-kms-key-
    //# arn.md#identifying-an-aws-kms-multi-region-key) this function MUST
    //# exit successfully.
    if (!multiRegionKeys.length)
        return;
    const multiRegionKeyIds = multiRegionKeys.map((mrk) => {
        const arn = (0, arn_parsing_1.parseAwsKmsKeyArn)(mrk);
        return arn ? arn.ResourceId : mrk;
    });
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If there are zero duplicate resource ids between the multi-region
    //# keys, this function MUST exit successfully
    if (new Set(multiRegionKeyIds).size === multiRegionKeys.length)
        return;
    //= compliance/framework/aws-kms/aws-kms-mrk-are-unique.txt#2.5
    //# If any duplicate multi-region resource ids exist, this function MUST
    //# yield an error that includes all identifiers with duplicate resource
    //# ids not only the first duplicate found.
    const duplicateMultiRegionIdentifiers = multiRegionKeyIds
        .map((mrk, i, a) => {
        if (a.indexOf(mrk) !== a.lastIndexOf(mrk))
            return multiRegionKeys[i];
        /* Postcondition: Remove non-duplicate multi-Region keys. */
        return false;
    })
        .filter((dup) => dup)
        .join(',');
    throw new Error(`Related multi-Region keys: ${duplicateMultiRegionIdentifiers} are not allowed.`);
}
exports.awsKmsMrkAreUnique = awsKmsMrkAreUnique;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzX2ttc19tcmtfYXJlX3VuaXF1ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9hd3Nfa21zX21ya19hcmVfdW5pcXVlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEMsK0NBQWdGO0FBRWhGLCtEQUErRDtBQUMvRCw0QkFBNEI7QUFDNUIsU0FBZ0Isa0JBQWtCLENBQUMsZ0JBQTBCO0lBQzNELE1BQU0sZUFBZSxHQUFHLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFLENBQ3BELElBQUEsMkNBQTZCLEVBQUMsQ0FBQyxDQUFDLENBQ2pDLENBQUE7SUFFRCwrREFBK0Q7SUFDL0Qsb0VBQW9FO0lBQ3BFLHNFQUFzRTtJQUN0RSxzQkFBc0I7SUFDdEIsSUFBSSxDQUFDLGVBQWUsQ0FBQyxNQUFNO1FBQUUsT0FBTTtJQUVuQyxNQUFNLGlCQUFpQixHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRTtRQUNwRCxNQUFNLEdBQUcsR0FBRyxJQUFBLCtCQUFpQixFQUFDLEdBQUcsQ0FBQyxDQUFBO1FBQ2xDLE9BQU8sR0FBRyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUE7SUFDbkMsQ0FBQyxDQUFDLENBQUE7SUFDRiwrREFBK0Q7SUFDL0QscUVBQXFFO0lBQ3JFLDhDQUE4QztJQUM5QyxJQUFJLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLENBQUMsSUFBSSxLQUFLLGVBQWUsQ0FBQyxNQUFNO1FBQUUsT0FBTTtJQUV0RSwrREFBK0Q7SUFDL0Qsd0VBQXdFO0lBQ3hFLHdFQUF3RTtJQUN4RSwyQ0FBMkM7SUFDM0MsTUFBTSwrQkFBK0IsR0FBRyxpQkFBaUI7U0FDdEQsR0FBRyxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRTtRQUNqQixJQUFJLENBQUMsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUM7WUFBRSxPQUFPLGVBQWUsQ0FBQyxDQUFDLENBQUMsQ0FBQTtRQUNwRSw0REFBNEQ7UUFDNUQsT0FBTyxLQUFLLENBQUE7SUFDZCxDQUFDLENBQUM7U0FDRCxNQUFNLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNwQixJQUFJLENBQUMsR0FBRyxDQUFDLENBQUE7SUFFWixNQUFNLElBQUksS0FBSyxDQUNiLDhCQUE4QiwrQkFBK0IsbUJBQW1CLENBQ2pGLENBQUE7QUFDSCxDQUFDO0FBcENELGdEQW9DQyJ9