/*-
 * SPDX-License-Identifier: BSD-2-Clause
 *
 * Copyright (c) 1998 The NetBSD Foundation, Inc. All rights reserved.
 * Copyright (c) 1998 Lennart Augustsson. All rights reserved.
 * Copyright (c) 2008 Hans Petter Selasky. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/stdint.h>
#include <sys/stddef.h>
#include <sys/param.h>
#include <sys/queue.h>
#include <sys/types.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/bus.h>
#include <sys/module.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/condvar.h>
#include <sys/sysctl.h>
#include <sys/sx.h>
#include <sys/unistd.h>
#include <sys/callout.h>
#include <sys/malloc.h>
#include <sys/priv.h>

#include <dev/usb/usb.h>
#include <dev/usb/usb_ioctl.h>
#include <dev/usb/usbdi.h>
#include "usbdevs.h"

#define	USB_DEBUG_VAR usb_debug
#include <dev/usb/usb_debug.h>
#include <dev/usb/usb_dynamic.h>

#include <dev/usb/quirk/usb_quirk.h>

MODULE_DEPEND(usb_quirk, usb, 1, 1, 1);
MODULE_VERSION(usb_quirk, 1);

#define	USB_DEV_QUIRKS_MAX 384
#define	USB_SUB_QUIRKS_MAX 8
#define	USB_QUIRK_ENVROOT "hw.usb.quirk."

struct usb_quirk_entry {
	uint16_t vid;
	uint16_t pid;
	uint16_t lo_rev;
	uint16_t hi_rev;
	uint16_t quirks[USB_SUB_QUIRKS_MAX];
};

static struct mtx usb_quirk_mtx;

#define	USB_QUIRK(v, p, ...) { \
  .vid = USB_VENDOR_##v, .pid = USB_PRODUCT_##v##_##p, .lo_rev = 0x0000, \
  .hi_rev = 0xffff, .quirks = { __VA_ARGS__ } \
}

/* Vendor only */
#define	USB_QUIRK_VO(v, ...) { \
  .vid = USB_VENDOR_##v, .pid = 0x0000, .lo_rev = 0x0000, .hi_rev = 0xffff, \
  .quirks = { UQ_MATCH_VENDOR_ONLY, __VA_ARGS__ } \
}

/* Specific revision(s) */
#define	USB_QUIRK_REV(v, p, l, h, ...) { \
  .vid = USB_VENDOR_##v, .pid = USB_PRODUCT_##v##_##p, .lo_rev = (l), \
  .hi_rev = (h), .quirks = { __VA_ARGS__ } \
}

static struct usb_quirk_entry usb_quirks[USB_DEV_QUIRKS_MAX] = {
	USB_QUIRK(ASUS, LCM, UQ_HID_IGNORE),
	USB_QUIRK_REV(INSIDEOUT, EDGEPORT4, 0x094, 0x094, UQ_SWAP_UNICODE),
	USB_QUIRK_REV(DALLAS, J6502, 0x0a2, 0x0a2, UQ_BAD_ADC),
	USB_QUIRK_REV(DALLAS, J6502, 0x0a2, 0x0a2, UQ_AU_NO_XU),
	USB_QUIRK_REV(ALTEC, ADA70, 0x103, 0x103, UQ_BAD_ADC),
	USB_QUIRK_REV(ALTEC, ASC495, 0x000, 0x000, UQ_BAD_AUDIO),
	USB_QUIRK_REV(QTRONIX, 980N, 0x110, 0x110, UQ_SPUR_BUT_UP),
	USB_QUIRK_REV(ALCOR2, KBD_HUB, 0x001, 0x001, UQ_SPUR_BUT_UP),
	USB_QUIRK_REV(MCT, HUB0100, 0x102, 0x102, UQ_BUS_POWERED),
	USB_QUIRK_REV(MCT, USB232, 0x102, 0x102, UQ_BUS_POWERED),
	USB_QUIRK_REV(TI, UTUSB41, 0x110, 0x110, UQ_POWER_CLAIM),
	USB_QUIRK_REV(TELEX, MIC1, 0x009, 0x009, UQ_AU_NO_FRAC),
	USB_QUIRK_REV(SILICONPORTALS, YAPPHONE, 0x100, 0x100, UQ_AU_INP_ASYNC),
	USB_QUIRK(LOGITECH, UN53B, UQ_NO_STRINGS),
	USB_QUIRK(LOGITECH, G510S, UQ_KBD_BOOTPROTO),
	USB_QUIRK(REALTEK, RTL8196EU, UQ_CFG_INDEX_1),
	USB_QUIRK(ELSA, MODEM1, UQ_CFG_INDEX_1),
	USB_QUIRK(PLANEX2, MZKUE150N, UQ_CFG_INDEX_1),
	USB_QUIRK(CISCOLINKSYS, USB3GIGV1, UQ_CFG_INDEX_1),
	/* Quirks for printer devices */
	USB_QUIRK(HP, 895C, UQ_BROKEN_BIDIR),
	USB_QUIRK(HP, 880C, UQ_BROKEN_BIDIR),
	USB_QUIRK(HP, 815C, UQ_BROKEN_BIDIR),
	USB_QUIRK(HP, 810C, UQ_BROKEN_BIDIR),
	USB_QUIRK(HP, 830C, UQ_BROKEN_BIDIR),
	USB_QUIRK(HP, 1220C, UQ_BROKEN_BIDIR),
	USB_QUIRK(XEROX, WCM15, UQ_BROKEN_BIDIR),
	/* Devices which should be ignored by uhid */
	USB_QUIRK(APC, UPS, UQ_HID_IGNORE),
	USB_QUIRK(APC, UPS1000, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6H375USB, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C550AVR, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C1250TWRK, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C1500TWRK, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C900UNV, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C100UNV, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C120UNV, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C800UNV, UQ_HID_IGNORE),
	USB_QUIRK(BELKIN, F6C1100UNV, UQ_HID_IGNORE),
	USB_QUIRK(CYBERPOWER, BC900D, UQ_HID_IGNORE),
	USB_QUIRK(CYBERPOWER, 1500CAVRLCD, UQ_HID_IGNORE),
	USB_QUIRK(CYBERPOWER, OR2200LCDRM2U, UQ_HID_IGNORE),
	USB_QUIRK(DELL2, VARIOUS_UPS, UQ_HID_IGNORE),
	USB_QUIRK(CYPRESS, SILVERSHIELD, UQ_HID_IGNORE),
	USB_QUIRK(DELORME, EARTHMATE, UQ_HID_IGNORE),
	USB_QUIRK(DREAMLINK, DL100B, UQ_HID_IGNORE),
	USB_QUIRK(MICROCHIP, PICOLCD20X2, UQ_HID_IGNORE),
	USB_QUIRK(MICROCHIP, PICOLCD4X20, UQ_HID_IGNORE),
	USB_QUIRK(LIEBERT, POWERSURE_PXT, UQ_HID_IGNORE),
	USB_QUIRK(LIEBERT2, PSI1000, UQ_HID_IGNORE),
	USB_QUIRK(LIEBERT2, POWERSURE_PSA, UQ_HID_IGNORE),
	USB_QUIRK(MGE, UPS1, UQ_HID_IGNORE),
	USB_QUIRK(MGE, UPS2, UQ_HID_IGNORE),
	USB_QUIRK(POWERCOM, IMPERIAL_SERIES, UQ_HID_IGNORE),
	USB_QUIRK(POWERCOM, SMART_KING_PRO, UQ_HID_IGNORE),
	USB_QUIRK(POWERCOM, WOW, UQ_HID_IGNORE),
	USB_QUIRK(POWERCOM, VANGUARD, UQ_HID_IGNORE),
	USB_QUIRK(POWERCOM, BLACK_KNIGHT_PRO, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, AVR550U, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, AVR750U, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, ECO550UPS, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, T750_INTL, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, RT_2200_INTL, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, OMNI1000LCD, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, OMNI900LCD, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, SMART_2200RMXL2U, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, UPS_3014, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, SU1500RTXL2UA, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, SU6000RT4U, UQ_HID_IGNORE),
	USB_QUIRK(TRIPPLITE2, SU1500RTXL2UA_2, UQ_HID_IGNORE),
	USB_QUIRK(APPLE, IPHONE, UQ_HID_IGNORE),
	USB_QUIRK(APPLE, IPHONE_3G, UQ_HID_IGNORE),
	USB_QUIRK(MEGATEC, UPS, UQ_HID_IGNORE),
	/* Devices which should be ignored by both ukbd and uhid */
	USB_QUIRK(CYPRESS, WISPY1A, UQ_KBD_IGNORE, UQ_HID_IGNORE),
	USB_QUIRK(METAGEEK, WISPY1B, UQ_KBD_IGNORE, UQ_HID_IGNORE),
	USB_QUIRK(METAGEEK, WISPY24X, UQ_KBD_IGNORE, UQ_HID_IGNORE),
	USB_QUIRK(METAGEEK2, WISPYDBX, UQ_KBD_IGNORE, UQ_HID_IGNORE),
	USB_QUIRK_REV(TENX, UAUDIO0, 0x0101, 0x0101, UQ_AUDIO_SWAP_LR),
	/* MS keyboards do weird things */
	USB_QUIRK(MICROSOFT, NATURAL4000, UQ_KBD_BOOTPROTO),
	USB_QUIRK(MICROSOFT, WLINTELLIMOUSE, UQ_MS_LEADING_BYTE),
	/* Quirk for Corsair Vengeance K60 keyboard */
	USB_QUIRK(CORSAIR, K60, UQ_KBD_BOOTPROTO),
	/* Quirk for Corsair Gaming K68 keyboard */
	USB_QUIRK(CORSAIR, K68, UQ_KBD_BOOTPROTO),
	/* Quirk for Corsair Vengeance K70 keyboard */
	USB_QUIRK(CORSAIR, K70, UQ_KBD_BOOTPROTO),
	/* Quirk for Corsair K70 RGB keyboard */
	USB_QUIRK(CORSAIR, K70_RGB, UQ_KBD_BOOTPROTO),
	/* Quirk for Corsair STRAFE Gaming keyboard */
	USB_QUIRK(CORSAIR, STRAFE, UQ_KBD_BOOTPROTO),
	USB_QUIRK(CORSAIR, STRAFE2, UQ_KBD_BOOTPROTO),
	/* Quirk for Kensington Slimblade Trackball */
	USB_QUIRK(KENSINGTON, SLIMBLADE, UQ_MS_VENDOR_BTN),
	/* umodem(4) device quirks */
	USB_QUIRK_REV(METRICOM, RICOCHET_GS, 0x100, 0x100, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK_REV(SANYO, SCP4900, 0x000, 0x000, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK_REV(MOTOROLA2, T720C, 0x001, 0x001, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK_REV(EICON, DIVA852, 0x100, 0x100, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK_REV(SIEMENS2, ES75, 0x000, 0x000, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(QUALCOMM, CDMA_MSM, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(QUALCOMM2, CDMA_MSM, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(CURITEL, UM150, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(CURITEL, UM175, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(VERTEX, VW110L, UQ_ASSUME_CM_OVER_DATA),
	USB_QUIRK(BALTECH, SMARTCARDREADER, UQ_IGNORE_CDC_CM),

	/* USB Mass Storage Class Quirks */
	USB_QUIRK_VO(ASAHIOPTICAL, UQ_MSC_NO_RS_CLEAR_UA),
	USB_QUIRK(ADDON, ATTACHE, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(ADDON, A256MB, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(ADDON, DISKPRO512, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(ADDONICS2, CABLE_205, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(AIPTEK, POCKETCAM3M, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ALCOR, UMCR_9361, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(APACER, HT202, UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(ASAHIOPTICAL, OPTIO230, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(ASAHIOPTICAL, OPTIO330, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(ATP, EUSB, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(BELKIN, USB2SCSI, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(CASIO, QV_DIGICAM, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(CCYU, ED1064, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(CENTURY, EX35QUAT, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK_REV(CREATIVE, NOMAD, 0x0001, 0xffff, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_READ_CAP_OFFBY1),
	USB_QUIRK(CREATIVE, STAGE_SE_MINI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(CYPRESS, XX6830XX, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(EMTEC, DANEELEC4GB, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(DESKNOTE, UCR_61S2B, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(DMI, CFSM_RW, UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(EMTEC, RUF2PS, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(EPSON, STYLUS_875DC, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(EPSON, STYLUS_895, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(FEIYA, 5IN1, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(FEIYA, ELANGO, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(FREECOM, DVD, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(FUJIPHOTO, MASS0100, UQ_MSC_FORCE_WIRE_CBI_I,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_RS_CLEAR_UA, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(GARMIN, DAKOTA20, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(GARMIN, FORERUNNER230, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(GARMIN, GPSMAP62S, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(GARMIN, EDGETOURINGPLUS, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(GARMIN, INSTINCTSOLAR, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(GENESYS, GL641USB2IDE, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP,
	    UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(GENESYS, GL641USB2IDE_2, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_FORCE_SHORT_INQ,
	    UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(GENESYS, GL641USB, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(GENESYS, GL641USB_2, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_WRONG_CSWSIG),
	USB_QUIRK(GENESYS, GL3220, UQ_MSC_NO_INQUIRY, UQ_MSC_NO_RS_CLEAR_UA),
	USB_QUIRK(HAGIWARA, FG, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(HAGIWARA, FGSM, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(HITACHI, DVDCAM_DZ_MV100A, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(HITACHI, DVDCAM_USB, UQ_MSC_FORCE_WIRE_CBI_I,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(HP, CDW4E, UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(HP, CDW8200, UQ_MSC_FORCE_WIRE_CBI_I, UQ_MSC_FORCE_PROTO_ATAPI,
	    UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_START_STOP),
	USB_QUIRK(HUAWEI, E3372_INIT, UQ_MSC_NO_INQUIRY, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(IMAGINATION, DBX1, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_WRONG_CSWSIG),
	USB_QUIRK(INSYSTEM, USBCABLE, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_ATAPI,
	    UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_START_STOP, UQ_MSC_ALT_IFACE_1),
	USB_QUIRK(INSYSTEM, ATAPI, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(INSYSTEM, STORAGE_V2, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(VIALABS, VL701, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(IODATA, IU_CD2, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(IODATA, DVR_UEH8, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(IOMEGA, ZIP100, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_TEST_UNIT_READY), /* XXX ZIP drives can also use ATAPI */
	USB_QUIRK(JMICRON, JMS566, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(JMICRON, JMS567, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(JMICRON, JM20337, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(KINGSTON, HYPERX3_0, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(KINGSTON, DATATRAVELER3_0, UQ_MSC_NO_PREVENT_ALLOW,
	    UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(KYOCERA, FINECAM_L3, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(KYOCERA, FINECAM_S3X, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(KYOCERA, FINECAM_S4, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(KYOCERA, FINECAM_S5, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(LACIE, HD, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(LEXAR, CF_READER, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(LEXAR, JUMPSHOT, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(LEXAR, JUMPDRIVE, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(LOGITEC, LDR_H443SU2, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(LOGITEC, LDR_H443U2, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(MELCO, DUBPXXG, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(MICROTECH, DPCM, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_START_STOP),
	USB_QUIRK(MICRON, REALSSD, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(MICROTECH, SCSIDB25, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(MICROTECH, SCSIHD50, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(MINOLTA, E223, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(MINOLTA, F300, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(MITSUMI, CDRRW, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(MOTOROLA2, E398, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_INQUIRY_EVPD, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK_VO(MPMAN, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(MSYSTEMS, DISKONKEY, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_GETMAXLUN,
	    UQ_MSC_NO_RS_CLEAR_UA),
	USB_QUIRK(MSYSTEMS, DISKONKEY2, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(MYSON, HEDEN, UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(NEODIO, ND3260, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_FORCE_SHORT_INQ),
	USB_QUIRK(NETAC, CF_CARD, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(NETAC, ONLYDISK, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(NETCHIP, CLIK_40, UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(NETCHIP, POCKETBOOK, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(NIKON, D300, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(NORELSYS, NS1081, UQ_MSC_NO_RS_CLEAR_UA, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(OLYMPUS, C1, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_WRONG_CSWSIG),
	USB_QUIRK(OLYMPUS, C700, UQ_MSC_NO_GETMAXLUN),
        /* Selected Olympus DSLR and EVIL models.  See ../usbdevs for more
	 * details.
	 *
	 * Not all quirks apply to all models.  The commented-out entries are
	 * correct for that model.
	 */
	USB_QUIRK(OLYMPUS, E_1, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_TEST_UNIT_READY,
	    UQ_MSC_NO_PREVENT_ALLOW, UQ_MSC_NO_SYNC_CACHE),
	/*
	 * Product code 0x118.
	 * USB_QUIRK(OLYMPUS, E_300, UQ_MSC_NO_GETMAXLUN,
	 *     UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_PREVENT_ALLOW,
         *     UQ_MSC_NO_SYNC_CACHE),
	 * USB_QUIRK(OLYMPUS, E_30, UQ_MSC_NO_GETMAXLUN,
	 *     UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_PREVENT_ALLOW,
         *     UQ_MSC_NO_SYNC_CACHE), */
	USB_QUIRK(OLYMPUS, E_330, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_TEST_UNIT_READY,
	    UQ_MSC_NO_PREVENT_ALLOW, UQ_MSC_NO_SYNC_CACHE, UQ_MSC_NO_START_STOP),
	USB_QUIRK(OLYMPUS, E_PM1, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_TEST_UNIT_READY,
	    UQ_MSC_NO_PREVENT_ALLOW, UQ_MSC_NO_SYNC_CACHE, UQ_MSC_NO_START_STOP),
        /* Product code 0x12e.
	 * USB_QUIRK(OLYMPUS, E_PM2, 0),
	 * USB_QUIRK(OLYMPUS, E_M1MarkII, UQ_MSC_NO_GETMAXLUN,
	 *     UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_PREVENT_ALLOW,
	 *     UQ_MSC_NO_SYNC_CACHE),
	 * USB_QUIRK(OLYMPUS, E_M5MarkIII, 0),
	 */
	USB_QUIRK(OLYMPUS, E_M1, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_TEST_UNIT_READY,
	    UQ_MSC_NO_PREVENT_ALLOW, UQ_MSC_NO_SYNC_CACHE, UQ_MSC_NO_START_STOP),
	USB_QUIRK(ONSPEC, SDS_HOTFIND_D, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(ONSPEC, CFMS_RW, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, CFSM_COMBO, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, CFSM_READER, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, CFSM_READER2, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, MDCFE_B_CF_READER, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, MDSM_B_READER, UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(ONSPEC, READER, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(ONSPEC, UCF100, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(ONSPEC2, IMAGEMATE_SDDR55, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(PANASONIC, KXL840AN, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(PANASONIC, KXLCB20AN, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(PANASONIC, KXLCB35AN, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(PANASONIC, LS120CAM, UQ_MSC_FORCE_PROTO_UFI),
	USB_QUIRK(PLEXTOR, 40_12_40U, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_TEST_UNIT_READY),
	USB_QUIRK(PNY, ATTACHE2, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_START_STOP),
	USB_QUIRK(PROLIFIC, PL2506, UQ_MSC_NO_SYNC_CACHE, UQ_MSC_NO_PREVENT_ALLOW),
	USB_QUIRK(SAMSUNG_TECHWIN, DIGIMAX_410, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(SANDISK, SDDR05A, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_READ_CAP_OFFBY1),
	USB_QUIRK(SANDISK, SDDR09, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_READ_CAP_OFFBY1, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(SANDISK, SDDR12, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_READ_CAP_OFFBY1),
	USB_QUIRK(SANDISK, SDCZ2_128, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(SANDISK, SDCZ2_256, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(SANDISK, SDCZ4_128, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(SANDISK, SDCZ4_256, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(SANDISK, SDCZ48_32, UQ_MSC_NO_SYNC_CACHE,
	    UQ_MSC_NO_TEST_UNIT_READY),
	USB_QUIRK(SANDISK, SDDR31, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_READ_CAP_OFFBY1),
	USB_QUIRK(SANDISK, IMAGEMATE_SDDR289, UQ_MSC_NO_SYNC_CACHE,
	    UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(SCANLOGIC, SL11R, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(SHUTTLE, EUSB, UQ_MSC_FORCE_WIRE_CBI_I, UQ_MSC_FORCE_PROTO_ATAPI,
	    UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_START_STOP, UQ_MSC_SHUTTLE_INIT),
	USB_QUIRK(SHUTTLE, CDRW, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(SHUTTLE, CF, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(SHUTTLE, EUSBATAPI, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(SHUTTLE, EUSBCFSM, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(SHUTTLE, EUSCSI, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(SHUTTLE, HIFD, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(SHUTTLE, SDDR09, UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(SHUTTLE, ZIOMMC, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(SIGMATEL, I_BEAD100, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_SHUTTLE_INIT),
	USB_QUIRK(SIIG, WINTERREADER, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(SKANHEX, MD_7425, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(SKANHEX, SX_520Z, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK_REV(SONY, HANDYCAM, 0x0500, 0x0500, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_RBC, UQ_MSC_RBC_PAD_TO_12),
	USB_QUIRK(SONY, CLIE_40_MS, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK_REV(SONY, DSC, 0x0500, 0x0500, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_RBC, UQ_MSC_RBC_PAD_TO_12),
	USB_QUIRK_REV(SONY, DSC, 0x0600, 0x0600, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_RBC, UQ_MSC_RBC_PAD_TO_12),
	USB_QUIRK(SONY, DSC, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(SONY, HANDYCAM, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(SONY, MSC, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(SONY, MS_MSC_U03, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_UFI),
	USB_QUIRK(SONY, MS_NW_MS7, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(SONY, MS_PEG_N760C, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_INQUIRY),
	USB_QUIRK(SONY, MSACUS1, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_GETMAXLUN),
	USB_QUIRK(SONY, PORTABLE_HDD_V2, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(STMICRO, ST72682, UQ_MSC_NO_PREVENT_ALLOW),
	USB_QUIRK(SUPERTOP, IDE, UQ_MSC_IGNORE_RESIDUE, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(SUPERTOP, FLASHDRIVE, UQ_MSC_NO_TEST_UNIT_READY,
	    UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(TAUGA, CAMERAMATE, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(TEAC, FD05PUB, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_UFI),
	USB_QUIRK(TECLAST, TLC300, UQ_MSC_NO_TEST_UNIT_READY, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(TREK, MEMKEY, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_INQUIRY),
	USB_QUIRK(TREK, THUMBDRIVE_8MB, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(TRUMPION, C3310, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_UFI),
	USB_QUIRK(TRUMPION, MP3, UQ_MSC_FORCE_PROTO_RBC),
	USB_QUIRK(TRUMPION, T33520, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(TWINMOS, MDIV, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(VIA, USB2IDEBRIDGE, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_FORCE_PROTO_SCSI, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(VIVITAR, 35XX, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_INQUIRY),
	USB_QUIRK(WESTERN, COMBO, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(WESTERN, EXTHDD, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(WESTERN, MYBOOK, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_INQUIRY_EVPD, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_00, UQ_MSC_FORCE_SHORT_INQ),
	USB_QUIRK(WESTERN, MYPASSPORT_01, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_02, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_03, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_04, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_05, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_06, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_07, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_08, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_09, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_10, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORT_11, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_00, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_01, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_02, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_03, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_04, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_05, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_06, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_07, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_08, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTES_09, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(WESTERN, MYPASSPORTUL_00, UQ_MSC_NO_TEST_UNIT_READY),
	USB_QUIRK(WINMAXGROUP, FLASH64MC, UQ_MSC_FORCE_WIRE_BBB,
	    UQ_MSC_NO_INQUIRY, UQ_MSC_FORCE_PROTO_SCSI),
	USB_QUIRK(YANO, FW800HD, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_FORCE_SHORT_INQ, UQ_MSC_NO_START_STOP, UQ_MSC_IGNORE_RESIDUE),
	USB_QUIRK(YANO, U640MO, UQ_MSC_FORCE_WIRE_CBI_I,
	    UQ_MSC_FORCE_PROTO_ATAPI, UQ_MSC_FORCE_SHORT_INQ),
	USB_QUIRK_REV(YEDATA, FLASHBUSTERU, 0x0000, 0x007F, UQ_MSC_FORCE_WIRE_CBI,
	    UQ_MSC_FORCE_PROTO_UFI, UQ_MSC_NO_RS_CLEAR_UA, UQ_MSC_FLOPPY_SPEED,
	    UQ_MSC_NO_TEST_UNIT_READY),
	USB_QUIRK_REV(YEDATA, FLASHBUSTERU, 0x0080, 0x0080,
	    UQ_MSC_FORCE_WIRE_CBI_I, UQ_MSC_FORCE_PROTO_UFI,
	    UQ_MSC_NO_RS_CLEAR_UA, UQ_MSC_FLOPPY_SPEED,
	    UQ_MSC_NO_TEST_UNIT_READY),
	USB_QUIRK_REV(YEDATA, FLASHBUSTERU, 0x0081, 0xFFFF,
	    UQ_MSC_FORCE_WIRE_CBI_I, UQ_MSC_FORCE_PROTO_UFI,
	    UQ_MSC_NO_RS_CLEAR_UA, UQ_MSC_FLOPPY_SPEED),
	USB_QUIRK(ZORAN, EX20DSC, UQ_MSC_FORCE_WIRE_CBI, UQ_MSC_FORCE_PROTO_ATAPI),
	USB_QUIRK(MEIZU, M6_SL, UQ_MSC_FORCE_WIRE_BBB, UQ_MSC_FORCE_PROTO_SCSI,
	    UQ_MSC_NO_INQUIRY, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(TOSHIBA, TRANSMEMORY, UQ_MSC_NO_SYNC_CACHE,
	    UQ_MSC_NO_PREVENT_ALLOW),
	USB_QUIRK(VIALABS, USB30SATABRIDGE, UQ_MSC_NO_SYNC_CACHE),
	USB_QUIRK(QUALCOMMINC, ZTE_MF730M, UQ_MSC_NO_GETMAXLUN,
	    UQ_MSC_NO_INQUIRY, UQ_CFG_INDEX_0),
	USB_QUIRK(SMART2, G2MEMKEY, UQ_MSC_NO_INQUIRY),
	USB_QUIRK_REV(RALINK, RT_STOR, 0x0001, 0x0001, UQ_MSC_IGNORE),
	USB_QUIRK(REALTEK, RTW8821CU_CD, UQ_MSC_IGNORE),
	/* Non-standard USB MIDI devices */
	USB_QUIRK(ROLAND, UM1, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SC8850, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SD90, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UM880N, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UA100, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UM4, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, U8, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UM2, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SC8820, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, PC300, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SK500, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SCD70, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UM550, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SD20, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, SD80, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, UA700, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(ROLAND, PCR300, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(EGO, M4U, UQ_SINGLE_CMD_MIDI),
	USB_QUIRK(LOGILINK, U2M, UQ_SINGLE_CMD_MIDI),
	USB_QUIRK(MEDELI, DD305, UQ_SINGLE_CMD_MIDI, UQ_MATCH_VENDOR_ONLY),
	USB_QUIRK(REDOCTANE, GHMIDI, UQ_SINGLE_CMD_MIDI),
	USB_QUIRK(TEXTECH, U2M_1, UQ_SINGLE_CMD_MIDI),
	USB_QUIRK(TEXTECH, U2M_2, UQ_SINGLE_CMD_MIDI),
	USB_QUIRK(WCH2, U2M, UQ_SINGLE_CMD_MIDI),

	/* Non-standard USB AUDIO devices */
	USB_QUIRK(MAUDIO, FASTTRACKULTRA, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(MAUDIO, FASTTRACKULTRA8R, UQ_AU_VENDOR_CLASS),
	USB_QUIRK(CMEDIA, CM6206, UQ_AU_SET_SPDIF_CM6206),
	USB_QUIRK(PLOYTEC, SPL_CRIMSON_1, UQ_CFG_INDEX_1),
	USB_QUIRK(ROLAND, UA25EX_AD, UQ_AU_VENDOR_CLASS),

	/* DYMO LabelManager Pnp */
	USB_QUIRK(DYMO, LABELMANAGERPNP, UQ_MSC_DYMO_EJECT),

	/* Holtek USB gaming keyboard */
	USB_QUIRK(HOLTEK, F85, UQ_KBD_BOOTPROTO),

	/* This works much better with if_cdce than if_ure */
	USB_QUIRK(LENOVO, TBT3LAN, UQ_CFG_INDEX_1),
};
#undef USB_QUIRK_VO
#undef USB_QUIRK_REV
#undef USB_QUIRK

static const char *usb_quirk_str[USB_QUIRK_MAX] = {
	[UQ_NONE]		= "UQ_NONE",
	[UQ_MATCH_VENDOR_ONLY]	= "UQ_MATCH_VENDOR_ONLY",
	[UQ_AUDIO_SWAP_LR]	= "UQ_AUDIO_SWAP_LR",
	[UQ_AU_INP_ASYNC]	= "UQ_AU_INP_ASYNC",
	[UQ_AU_NO_FRAC]		= "UQ_AU_NO_FRAC",
	[UQ_AU_NO_XU]		= "UQ_AU_NO_XU",
	[UQ_BAD_ADC]		= "UQ_BAD_ADC",
	[UQ_BAD_AUDIO]		= "UQ_BAD_AUDIO",
	[UQ_BROKEN_BIDIR]	= "UQ_BROKEN_BIDIR",
	[UQ_BUS_POWERED]	= "UQ_BUS_POWERED",
	[UQ_HID_IGNORE]		= "UQ_HID_IGNORE",
	[UQ_KBD_IGNORE]		= "UQ_KBD_IGNORE",
	[UQ_KBD_BOOTPROTO]	= "UQ_KBD_BOOTPROTO",
	[UQ_UMS_IGNORE]		= "UQ_UMS_IGNORE",
	[UQ_MS_BAD_CLASS]	= "UQ_MS_BAD_CLASS",
	[UQ_MS_LEADING_BYTE]	= "UQ_MS_LEADING_BYTE",
	[UQ_MS_REVZ]		= "UQ_MS_REVZ",
	[UQ_MS_VENDOR_BTN]	= "UQ_MS_VENDOR_BTN",
	[UQ_NO_STRINGS]		= "UQ_NO_STRINGS",
	[UQ_POWER_CLAIM]	= "UQ_POWER_CLAIM",
	[UQ_SPUR_BUT_UP]	= "UQ_SPUR_BUT_UP",
	[UQ_SWAP_UNICODE]	= "UQ_SWAP_UNICODE",
	[UQ_CFG_INDEX_1]	= "UQ_CFG_INDEX_1",
	[UQ_CFG_INDEX_2]	= "UQ_CFG_INDEX_2",
	[UQ_CFG_INDEX_3]	= "UQ_CFG_INDEX_3",
	[UQ_CFG_INDEX_4]	= "UQ_CFG_INDEX_4",
	[UQ_CFG_INDEX_0]	= "UQ_CFG_INDEX_0",
	[UQ_ASSUME_CM_OVER_DATA]	= "UQ_ASSUME_CM_OVER_DATA",
	[UQ_IGNORE_CDC_CM]	= "UQ_IGNORE_CDC_CM",
	[UQ_MSC_NO_TEST_UNIT_READY]	= "UQ_MSC_NO_TEST_UNIT_READY",
	[UQ_MSC_NO_RS_CLEAR_UA]		= "UQ_MSC_NO_RS_CLEAR_UA",
	[UQ_MSC_NO_START_STOP]		= "UQ_MSC_NO_START_STOP",
	[UQ_MSC_NO_GETMAXLUN]		= "UQ_MSC_NO_GETMAXLUN",
	[UQ_MSC_NO_INQUIRY]		= "UQ_MSC_NO_INQUIRY",
	[UQ_MSC_NO_INQUIRY_EVPD]	= "UQ_MSC_NO_INQUIRY_EVPD",
	[UQ_MSC_NO_PREVENT_ALLOW]	= "UQ_MSC_NO_PREVENT_ALLOW",
	[UQ_MSC_NO_SYNC_CACHE]		= "UQ_MSC_NO_SYNC_CACHE",
	[UQ_MSC_SHUTTLE_INIT]		= "UQ_MSC_SHUTTLE_INIT",
	[UQ_MSC_ALT_IFACE_1]		= "UQ_MSC_ALT_IFACE_1",
	[UQ_MSC_FLOPPY_SPEED]		= "UQ_MSC_FLOPPY_SPEED",
	[UQ_MSC_IGNORE_RESIDUE]		= "UQ_MSC_IGNORE_RESIDUE",
	[UQ_MSC_WRONG_CSWSIG]		= "UQ_MSC_WRONG_CSWSIG",
	[UQ_MSC_RBC_PAD_TO_12]		= "UQ_MSC_RBC_PAD_TO_12",
	[UQ_MSC_READ_CAP_OFFBY1]	= "UQ_MSC_READ_CAP_OFFBY1",
	[UQ_MSC_FORCE_SHORT_INQ]	= "UQ_MSC_FORCE_SHORT_INQ",
	[UQ_MSC_FORCE_WIRE_BBB]		= "UQ_MSC_FORCE_WIRE_BBB",
	[UQ_MSC_FORCE_WIRE_CBI]		= "UQ_MSC_FORCE_WIRE_CBI",
	[UQ_MSC_FORCE_WIRE_CBI_I]	= "UQ_MSC_FORCE_WIRE_CBI_I",
	[UQ_MSC_FORCE_PROTO_SCSI]	= "UQ_MSC_FORCE_PROTO_SCSI",
	[UQ_MSC_FORCE_PROTO_ATAPI]	= "UQ_MSC_FORCE_PROTO_ATAPI",
	[UQ_MSC_FORCE_PROTO_UFI]	= "UQ_MSC_FORCE_PROTO_UFI",
	[UQ_MSC_FORCE_PROTO_RBC]	= "UQ_MSC_FORCE_PROTO_RBC",
	[UQ_MSC_IGNORE]			= "UQ_MSC_IGNORE",
	[UQ_MSC_EJECT_HUAWEI]		= "UQ_MSC_EJECT_HUAWEI",
	[UQ_MSC_EJECT_SIERRA]		= "UQ_MSC_EJECT_SIERRA",
	[UQ_MSC_EJECT_SCSIEJECT]	= "UQ_MSC_EJECT_SCSIEJECT",
	[UQ_MSC_EJECT_REZERO]		= "UQ_MSC_EJECT_REZERO",
	[UQ_MSC_EJECT_ZTESTOR]		= "UQ_MSC_EJECT_ZTESTOR",
	[UQ_MSC_EJECT_CMOTECH]		= "UQ_MSC_EJECT_CMOTECH",
	[UQ_MSC_EJECT_WAIT]		= "UQ_MSC_EJECT_WAIT",
	[UQ_MSC_EJECT_SAEL_M460]	= "UQ_MSC_EJECT_SAEL_M460",
	[UQ_MSC_EJECT_HUAWEISCSI]	= "UQ_MSC_EJECT_HUAWEISCSI",
	[UQ_MSC_EJECT_HUAWEISCSI2]	= "UQ_MSC_EJECT_HUAWEISCSI2",
	[UQ_MSC_EJECT_HUAWEISCSI3]	= "UQ_MSC_EJECT_HUAWEISCSI3",
	[UQ_MSC_EJECT_HUAWEISCSI4]	= "UQ_MSC_EJECT_HUAWEISCSI4",
	[UQ_MSC_EJECT_TCT]		= "UQ_MSC_EJECT_TCT",
	[UQ_BAD_MIDI]			= "UQ_BAD_MIDI",
	[UQ_AU_VENDOR_CLASS]		= "UQ_AU_VENDOR_CLASS",
	[UQ_SINGLE_CMD_MIDI]		= "UQ_SINGLE_CMD_MIDI",
	[UQ_MSC_DYMO_EJECT]		= "UQ_MSC_DYMO_EJECT",
	[UQ_AU_SET_SPDIF_CM6206]	= "UQ_AU_SET_SPDIF_CM6206",
	[UQ_WMT_IGNORE]			= "UQ_WMT_IGNORE",
};

/*------------------------------------------------------------------------*
 *	usb_quirkstr
 *
 * This function converts an USB quirk code into a string.
 *------------------------------------------------------------------------*/
static const char *
usb_quirkstr(uint16_t quirk)
{
	return ((quirk < USB_QUIRK_MAX && usb_quirk_str[quirk] != NULL) ?
	    usb_quirk_str[quirk] : "UQ_UNKNOWN");
}

/*------------------------------------------------------------------------*
 *	usb_strquirk
 *
 * This function converts a string into a USB quirk code.
 *
 * Returns:
 * Less than USB_QUIRK_MAX: Quirk code
 * Else: Quirk code not found
 *------------------------------------------------------------------------*/
static uint16_t
usb_strquirk(const char *str, size_t len)
{
	const char *quirk;
	uint16_t x;

	for (x = 0; x != USB_QUIRK_MAX; x++) {
		quirk = usb_quirkstr(x);
		if (strncmp(str, quirk, len) == 0 &&
		    quirk[len] == 0)
			break;
	}
	return (x);
}

/*------------------------------------------------------------------------*
 *	usb_test_quirk_by_info
 *
 * Returns:
 * 0: Quirk not found
 * Else: Quirk found
 *------------------------------------------------------------------------*/
static uint8_t
usb_test_quirk_by_info(const struct usbd_lookup_info *info, uint16_t quirk)
{
	uint16_t x;
	uint16_t y;

	if (quirk == UQ_NONE)
		goto done;

	USB_MTX_LOCK(&usb_quirk_mtx);

	for (x = 0; x != USB_DEV_QUIRKS_MAX; x++) {
		/* see if quirk information does not match */
		if ((usb_quirks[x].vid != info->idVendor) ||
		    (usb_quirks[x].lo_rev > info->bcdDevice) ||
		    (usb_quirks[x].hi_rev < info->bcdDevice)) {
			continue;
		}
		/* see if quirk only should match vendor ID */
		if (usb_quirks[x].pid != info->idProduct) {
			if (usb_quirks[x].pid != 0)
				continue;

			for (y = 0; y != USB_SUB_QUIRKS_MAX; y++) {
				if (usb_quirks[x].quirks[y] == UQ_MATCH_VENDOR_ONLY)
					break;
			}
			if (y == USB_SUB_QUIRKS_MAX)
				continue;
		}
		/* lookup quirk */
		for (y = 0; y != USB_SUB_QUIRKS_MAX; y++) {
			if (usb_quirks[x].quirks[y] == quirk) {
				USB_MTX_UNLOCK(&usb_quirk_mtx);
				DPRINTF("Found quirk '%s'.\n", usb_quirkstr(quirk));
				return (1);
			}
		}
	}
	USB_MTX_UNLOCK(&usb_quirk_mtx);
done:
	return (0);			/* no quirk match */
}

static struct usb_quirk_entry *
usb_quirk_get_entry(uint16_t vid, uint16_t pid,
    uint16_t lo_rev, uint16_t hi_rev, uint8_t do_alloc)
{
	uint16_t x;

	USB_MTX_ASSERT(&usb_quirk_mtx, MA_OWNED);

	if ((vid | pid | lo_rev | hi_rev) == 0) {
		/* all zero - special case */
		return (usb_quirks + USB_DEV_QUIRKS_MAX - 1);
	}
	/* search for an existing entry */
	for (x = 0; x != USB_DEV_QUIRKS_MAX; x++) {
		/* see if quirk information does not match */
		if ((usb_quirks[x].vid != vid) ||
		    (usb_quirks[x].pid != pid) ||
		    (usb_quirks[x].lo_rev != lo_rev) ||
		    (usb_quirks[x].hi_rev != hi_rev)) {
			continue;
		}
		return (usb_quirks + x);
	}

	if (do_alloc == 0) {
		/* no match */
		return (NULL);
	}
	/* search for a free entry */
	for (x = 0; x != USB_DEV_QUIRKS_MAX; x++) {
		/* see if quirk information does not match */
		if ((usb_quirks[x].vid |
		    usb_quirks[x].pid |
		    usb_quirks[x].lo_rev |
		    usb_quirks[x].hi_rev) != 0) {
			continue;
		}
		usb_quirks[x].vid = vid;
		usb_quirks[x].pid = pid;
		usb_quirks[x].lo_rev = lo_rev;
		usb_quirks[x].hi_rev = hi_rev;

		return (usb_quirks + x);
	}

	/* no entry found */
	return (NULL);
}

/*------------------------------------------------------------------------*
 *	usb_quirk_ioctl - handle quirk IOCTLs
 *
 * Returns:
 * 0: Success
 * Else: Failure
 *------------------------------------------------------------------------*/
static int
usb_quirk_ioctl(unsigned long cmd, caddr_t data,
    int fflag, struct thread *td)
{
	struct usb_gen_quirk *pgq;
	struct usb_quirk_entry *pqe;
	uint32_t x;
	uint32_t y;
	int err;

	switch (cmd) {
	case USB_DEV_QUIRK_GET:
		pgq = (void *)data;
		x = pgq->index % USB_SUB_QUIRKS_MAX;
		y = pgq->index / USB_SUB_QUIRKS_MAX;
		if (y >= USB_DEV_QUIRKS_MAX) {
			return (EINVAL);
		}
		USB_MTX_LOCK(&usb_quirk_mtx);
		/* copy out data */
		pgq->vid = usb_quirks[y].vid;
		pgq->pid = usb_quirks[y].pid;
		pgq->bcdDeviceLow = usb_quirks[y].lo_rev;
		pgq->bcdDeviceHigh = usb_quirks[y].hi_rev;
		strlcpy(pgq->quirkname,
		    usb_quirkstr(usb_quirks[y].quirks[x]),
		    sizeof(pgq->quirkname));
		USB_MTX_UNLOCK(&usb_quirk_mtx);
		return (0);		/* success */

	case USB_QUIRK_NAME_GET:
		pgq = (void *)data;
		x = pgq->index;
		if (x >= USB_QUIRK_MAX) {
			return (EINVAL);
		}
		strlcpy(pgq->quirkname,
		    usb_quirkstr(x), sizeof(pgq->quirkname));
		return (0);		/* success */

	case USB_DEV_QUIRK_ADD:
		pgq = (void *)data;

		/* check privileges */
		err = priv_check(curthread, PRIV_DRIVER);
		if (err) {
			return (err);
		}
		/* convert quirk string into numerical */
		for (y = 0; y != USB_DEV_QUIRKS_MAX; y++) {
			if (strcmp(pgq->quirkname, usb_quirkstr(y)) == 0) {
				break;
			}
		}
		if (y == USB_DEV_QUIRKS_MAX) {
			return (EINVAL);
		}
		if (y == UQ_NONE) {
			return (EINVAL);
		}
		USB_MTX_LOCK(&usb_quirk_mtx);
		pqe = usb_quirk_get_entry(pgq->vid, pgq->pid,
		    pgq->bcdDeviceLow, pgq->bcdDeviceHigh, 1);
		if (pqe == NULL) {
			USB_MTX_UNLOCK(&usb_quirk_mtx);
			return (EINVAL);
		}
		for (x = 0; x != USB_SUB_QUIRKS_MAX; x++) {
			if (pqe->quirks[x] == UQ_NONE) {
				pqe->quirks[x] = y;
				break;
			}
		}
		USB_MTX_UNLOCK(&usb_quirk_mtx);
		if (x == USB_SUB_QUIRKS_MAX) {
			return (ENOMEM);
		}
		return (0);		/* success */

	case USB_DEV_QUIRK_REMOVE:
		pgq = (void *)data;
		/* check privileges */
		err = priv_check(curthread, PRIV_DRIVER);
		if (err) {
			return (err);
		}
		/* convert quirk string into numerical */
		for (y = 0; y != USB_DEV_QUIRKS_MAX; y++) {
			if (strcmp(pgq->quirkname, usb_quirkstr(y)) == 0) {
				break;
			}
		}
		if (y == USB_DEV_QUIRKS_MAX) {
			return (EINVAL);
		}
		if (y == UQ_NONE) {
			return (EINVAL);
		}
		USB_MTX_LOCK(&usb_quirk_mtx);
		pqe = usb_quirk_get_entry(pgq->vid, pgq->pid,
		    pgq->bcdDeviceLow, pgq->bcdDeviceHigh, 0);
		if (pqe == NULL) {
			USB_MTX_UNLOCK(&usb_quirk_mtx);
			return (EINVAL);
		}
		for (x = 0; x != USB_SUB_QUIRKS_MAX; x++) {
			if (pqe->quirks[x] == y) {
				pqe->quirks[x] = UQ_NONE;
				break;
			}
		}
		if (x == USB_SUB_QUIRKS_MAX) {
			USB_MTX_UNLOCK(&usb_quirk_mtx);
			return (ENOMEM);
		}
		for (x = 0; x != USB_SUB_QUIRKS_MAX; x++) {
			if (pqe->quirks[x] != UQ_NONE) {
				break;
			}
		}
		if (x == USB_SUB_QUIRKS_MAX) {
			/* all quirk entries are unused - release */
			memset(pqe, 0, sizeof(*pqe));
		}
		USB_MTX_UNLOCK(&usb_quirk_mtx);
		return (0);		/* success */

	default:
		break;
	}
	return (ENOIOCTL);
}

/*------------------------------------------------------------------------*
 *	usb_quirk_strtou16
 *
 * Helper function to scan a 16-bit integer.
 *------------------------------------------------------------------------*/
static uint16_t
usb_quirk_strtou16(const char **pptr, const char *name, const char *what)
{
	unsigned long value;
	char *end;

	value = strtoul(*pptr, &end, 0);
	if (value > 65535 || *pptr == end || (*end != ' ' && *end != '\t')) {
		printf("%s: %s 16-bit %s value set to zero\n",
		    name, what, *end == 0 ? "incomplete" : "invalid");
		return (0);
	}
	*pptr = end + 1;
	return ((uint16_t)value);
}

/*------------------------------------------------------------------------*
 *	usb_quirk_add_entry_from_str
 *
 * Add a USB quirk entry from string.
 *     "VENDOR PRODUCT LO_REV HI_REV QUIRK[,QUIRK[,...]]"
 *------------------------------------------------------------------------*/
static void
usb_quirk_add_entry_from_str(const char *name, const char *env)
{
	struct usb_quirk_entry entry = { };
	struct usb_quirk_entry *new;
	uint16_t quirk_idx;
	uint16_t quirk;
	const char *end;

	/* check for invalid environment variable */
	if (name == NULL || env == NULL)
		return;

	if (bootverbose)
		printf("Adding USB QUIRK '%s' = '%s'\n", name, env);

	/* parse device information */
	entry.vid = usb_quirk_strtou16(&env, name, "Vendor ID");
	entry.pid = usb_quirk_strtou16(&env, name, "Product ID");
	entry.lo_rev = usb_quirk_strtou16(&env, name, "Low revision");
	entry.hi_rev = usb_quirk_strtou16(&env, name, "High revision");

	/* parse quirk information */
	quirk_idx = 0;
	while (*env != 0 && quirk_idx != USB_SUB_QUIRKS_MAX) {
		/* skip whitespace before quirks */
		while (*env == ' ' || *env == '\t')
			env++;

		/* look for quirk separation character */
		end = strchr(env, ',');
		if (end == NULL)
			end = env + strlen(env);

		/* lookup quirk in string table */
		quirk = usb_strquirk(env, end - env);
		if (quirk < USB_QUIRK_MAX) {
			entry.quirks[quirk_idx++] = quirk;
		} else {
			printf("%s: unknown USB quirk '%.*s' (skipped)\n",
			    name, (int)(end - env), env);
		}
		env = end;

		/* skip quirk delimiter, if any */
		if (*env != 0)
			env++;
	}

	/* register quirk */
	if (quirk_idx != 0) {
		if (*env != 0) {
			printf("%s: Too many USB quirks, only %d allowed!\n",
			    name, USB_SUB_QUIRKS_MAX);
		}
		USB_MTX_LOCK(&usb_quirk_mtx);
		new = usb_quirk_get_entry(entry.vid, entry.pid,
		    entry.lo_rev, entry.hi_rev, 1);
		if (new == NULL)
			printf("%s: USB quirks table is full!\n", name);
		else
			memcpy(new->quirks, entry.quirks, sizeof(entry.quirks));
		USB_MTX_UNLOCK(&usb_quirk_mtx);
	} else {
		printf("%s: No USB quirks found!\n", name);
	}
}

static void
usb_quirk_init(void *arg)
{
	char envkey[sizeof(USB_QUIRK_ENVROOT) + 2];	/* 2 digits max, 0 to 99 */
	int i;

	/* initialize mutex */
	mtx_init(&usb_quirk_mtx, "USB quirk", NULL, MTX_DEF);

	/* look for quirks defined by the environment variable */
	for (i = 0; i != 100; i++) {
		snprintf(envkey, sizeof(envkey), USB_QUIRK_ENVROOT "%d", i);

		/* Stop at first undefined var */
		if (!testenv(envkey))
			break;

		/* parse environment variable */
		usb_quirk_add_entry_from_str(envkey, kern_getenv(envkey));
	}

	/* register our function */
	usb_test_quirk_p = &usb_test_quirk_by_info;
	usb_quirk_ioctl_p = &usb_quirk_ioctl;
}

static void
usb_quirk_uninit(void *arg)
{
	usb_quirk_unload(arg);

	/* destroy mutex */
	mtx_destroy(&usb_quirk_mtx);
}

SYSINIT(usb_quirk_init, SI_SUB_LOCK, SI_ORDER_FIRST, usb_quirk_init, NULL);
SYSUNINIT(usb_quirk_uninit, SI_SUB_LOCK, SI_ORDER_ANY, usb_quirk_uninit, NULL);
